import React, {ReactElement, useRef, useEffect} from 'react';
import B from 'bem-cn-lite';

import IconGlyph from '../../interfaces/components/IconGlyph';
import ISuggestData from '../../interfaces/components/suggest/ISuggestData';
import ISuggestElement from '../../interfaces/components/suggest/ISuggestElement';

import useGetData from '../useGetData';
import typedMemo from '../typedMemo';

import Icon from '../Icon/Icon';
import Input from '../Input/Input';
import ModalHistoryWrapper from '../ModalHistoryWrapper/ModalHistoryWrapper';
import GlobalPopup from '../GlobalPopup';
import WrapSubstring from '../Wrap/WrapSubstring';

const b = B('SuggestMobile');

interface ISuggestMobile<SuggestElement extends ISuggestElement> {
    value: string;
    header: string;
    onClose: () => void;
    onChange: (value: string) => void;
    getData: (
        value: string,
    ) => ISuggestData<SuggestElement> | Promise<ISuggestData<SuggestElement>>;
    onClickElement: (element: SuggestElement) => void;

    placeholder?: string;
    limit?: number; // Максимальное количество отображаемых саджестов
    noDataText?: string; // Текст для случая, когда функция для получения данных вернула пустой результат
    regExpForHighlightingElements?: RegExp; // Регулярка для выделения совпавшего текста в саджестах
}

export default typedMemo(SuggestMobile);

function SuggestMobile<SuggestElement extends ISuggestElement>({
    value,
    header,
    onClose,
    onChange,
    getData,
    onClickElement,

    placeholder,
    limit = 20,
    noDataText,
    regExpForHighlightingElements,
}: ISuggestMobile<SuggestElement>): ReactElement {
    const suggestData = useGetData(value, getData);

    const inputRef = useRef<HTMLInputElement | null>(null);

    const elements = suggestData?.elements || [];

    useEffect(() => {
        const {current: input} = inputRef;

        if (input) {
            setTimeout(() => {
                // Почему-то без setTimeout не работает
                const lengthValue = input.value.length;

                input.focus();
                input.setSelectionRange(lengthValue, lengthValue);
            }, 1);
        }
    }, []);

    return (
        <ModalHistoryWrapper onClose={onClose}>
            <GlobalPopup>
                <div className={b()}>
                    <div className={b('header')}>
                        <span className={b('headerText')}>{header}</span>

                        <Icon
                            className={b('headerIcon')}
                            glyph={IconGlyph.cross}
                            flexShrinkZero
                            onClick={onClose}
                        />
                    </div>

                    <Input
                        className={b('input')}
                        value={value}
                        onChange={onChange}
                        placeholder={placeholder}
                        widthMod="100"
                        autoComplete={false}
                        autoCorrect={false}
                        withSearchIcon
                        withReset
                        inputRef={inputRef}
                    />

                    <div className={b('elements')}>
                        {elements.slice(0, limit).map(element => {
                            const {title, subtitle} = element;

                            return (
                                <div
                                    className={b('element', {
                                        withoutSubtitle: !subtitle,
                                    })}
                                    key={`${title}--${subtitle}`}
                                    // eslint-disable-next-line react/jsx-no-bind
                                    onClick={() => onClickElement(element)}
                                >
                                    <div className={b('elementTitle')}>
                                        {regExpForHighlightingElements ? (
                                            <WrapSubstring
                                                str={title}
                                                regExp={
                                                    regExpForHighlightingElements
                                                }
                                                className={b('bold')}
                                            />
                                        ) : (
                                            title
                                        )}
                                    </div>

                                    {subtitle && (
                                        <div className={b('elementSubtitle')}>
                                            {regExpForHighlightingElements ? (
                                                <WrapSubstring
                                                    str={subtitle}
                                                    regExp={
                                                        regExpForHighlightingElements
                                                    }
                                                    className={b('bold')}
                                                />
                                            ) : (
                                                subtitle
                                            )}
                                        </div>
                                    )}
                                </div>
                            );
                        })}

                        {elements.length === 0 && (
                            <div className={b('noDataText')}>{noDataText}</div>
                        )}
                    </div>
                </div>
            </GlobalPopup>
        </ModalHistoryWrapper>
    );
}
