import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';

import {TransportType} from '../../lib/transportType';

import ISegment from '../../interfaces/segment/ISegment';
import IStateCurrencies from '../../interfaces/state/IStateCurrencies';
import ISuburbanCategoryItem from '../../interfaces/segment/ISuburbanCategoryItem';
import ISegmentTariffClass from '../../interfaces/segment/ISegmentTariffClass';
import ITransfer from '../../interfaces/transfer/ITransfer';
import ITransferSegment from '../../interfaces/transfer/ITransferSegment';

import convertPrice from '../../lib/currencies/convertPrice';
import getSeatsTitle from '../../lib/thread/getSeatsTitle';

import Price from '../Price/Price';
import SegmentBuyLink from '../SegmentBuyLink';

import keyset from '../../i18n/tariffs-classes';

const b = B('TariffsListItem');

interface ITariffsListItemProps {
    segment: ISegment | ITransfer | ITransferSegment;
    tariffClass: ISuburbanCategoryItem | ISegmentTariffClass;

    classKey?: string;
    onClick?: Function;
    currencies?: IStateCurrencies;
    pageType?: string;
    viewMode?: 'detailed';
}

export default class TariffsListItem extends PureComponent<
    ITariffsListItemProps,
    {}
> {
    getClassTitle(): React.ReactElement | null {
        const {segment, tariffClass, classKey, viewMode} = this.props;
        const {
            transport: {code: transportType},
        } = segment;

        if (
            transportType === TransportType.suburban &&
            'isMain' in tariffClass &&
            !tariffClass.isMain
        ) {
            return null;
        }

        const isDetailed = viewMode === 'detailed';

        const classTitle =
            ('title' in tariffClass && tariffClass.title) ||
            (classKey && keyset.get(classKey)) ||
            '';
        const classIsAbbr = classTitle.length <= 2;

        const [shortClass, fullClass] = classIsAbbr
            ? [classTitle, null]
            : [classTitle.substr(0, 1), classTitle.substr(1)];

        return (
            <div className={b('tariff')}>
                <span className={b('tariffClass')}>
                    {shortClass && (
                        <abbr
                            className={b('shortClass', {hint: !classIsAbbr})}
                            aria-label={classTitle}
                        >
                            {shortClass}
                        </abbr>
                    )}
                    {fullClass && (
                        <span className={b('fullClass')}>{fullClass}</span>
                    )}
                </span>
                {'seats' in tariffClass &&
                    tariffClass.seats &&
                    (tariffClass.seats > 1 ||
                        transportType === TransportType.train) &&
                    (shortClass || fullClass) && (
                        <span className={b('seats')}>
                            {`${isDetailed ? '' : ': '}${tariffClass.seats} ${
                                isDetailed
                                    ? getSeatsTitle(tariffClass.seats)
                                    : ''
                            }`}
                        </span>
                    )}
            </div>
        );
    }

    render(): React.ReactElement {
        const {onClick, segment, tariffClass, currencies, viewMode} =
            this.props;
        const {
            transport: {code: transportType},
        } = segment;

        return (
            <li className={b({viewMode})}>
                <SegmentBuyLink
                    className={b('link', {
                        clickable: typeof onClick === 'function',
                    })}
                    onClick={onClick}
                    segment={segment}
                    tariffClass={tariffClass}
                >
                    <div className={b('classTitle')}>
                        {this.getClassTitle()}
                    </div>

                    <Price
                        className={b('price')}
                        from={transportType !== TransportType.suburban}
                        price={convertPrice(tariffClass.price, currencies)}
                    />
                </SegmentBuyLink>
            </li>
        );
    }
}
