import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';

import {THREAD_PAGE_NAME} from '../../routes/thread';

import {TransportType} from '../../lib/transportType';
import ISegment from '../../interfaces/segment/ISegment';
import ISuburbanCategoryItem from '../../interfaces/segment/ISuburbanCategoryItem';
import ISegmentTariffClass from '../../interfaces/segment/ISegmentTariffClass';
import IStateCurrencies from '../../interfaces/state/IStateCurrencies';
import ITransfer from '../../interfaces/transfer/ITransfer';
import ITransferSegment from '../../interfaces/transfer/ITransferSegment';

import noop from '../../lib/noop';
import convertPrice from '../../lib/currencies/convertPrice';
import getShortTariffClassName from '../../lib/segments/getShortTariffClassName';

import Price from '../Price/Price';
import SegmentBuyLink from '../SegmentBuyLink';

const b = B('TariffsListItem');

interface ITariffsListItemProps {
    segment: ISegment | ITransfer | ITransferSegment;
    tariffClass: ISuburbanCategoryItem | ISegmentTariffClass;

    classKey?: string;
    onClick?: Function;
    currencies?: IStateCurrencies;
    pageType?: string;
}

export default class TariffsListItem extends PureComponent<
    ITariffsListItemProps,
    {}
> {
    static defaultProps = {
        onClick: noop,
    };

    getClassTitle(): React.ReactElement | null {
        const {segment, tariffClass, classKey, pageType} = this.props;
        const {
            transport: {code: transportType},
        } = segment;

        if (transportType === TransportType.plane) {
            return null;
        }

        if (
            transportType === TransportType.suburban &&
            'isMain' in tariffClass &&
            !tariffClass.isMain
        ) {
            return null;
        }

        const classTitle =
            ('title' in tariffClass && tariffClass.title) ||
            getShortTariffClassName(classKey, pageType === THREAD_PAGE_NAME) ||
            '';

        return (
            <span
                className={b('classAndSeats', {
                    upper: pageType === THREAD_PAGE_NAME,
                })}
            >
                {`${classTitle}${
                    'seats' in tariffClass && tariffClass.seats
                        ? ` ${tariffClass.seats}`
                        : ''
                }`}
            </span>
        );
    }

    render(): React.ReactElement {
        const {onClick, segment, tariffClass, currencies, pageType} =
            this.props;

        const from =
            'severalPrices' in tariffClass &&
            tariffClass.severalPrices &&
            pageType !== THREAD_PAGE_NAME;

        return (
            <li className={b({pageType})}>
                <SegmentBuyLink
                    className={b('link')}
                    onClick={onClick}
                    segment={segment}
                    tariffClass={tariffClass}
                >
                    {this.getClassTitle()}

                    <Price
                        className={b('price', {
                            smaller: pageType === THREAD_PAGE_NAME,
                        })}
                        from={from}
                        price={convertPrice(tariffClass.price, currencies)}
                    />
                </SegmentBuyLink>
            </li>
        );
    }
}
