import React from 'react';
import {Moment} from 'moment';

import ISegment from '../../interfaces/segment/ISegment';
import ISearchContext from '../../interfaces/state/search/ISearchContext';

import {isCity} from '../../lib/point/pointType';
import getThreadUrl from '../../lib/url/getThreadUrl';
import {dayMatchesMask, getMaskValue} from '../../lib/date/mask';

import {useTld} from '../../hooks/useTld';
import {useLang} from '../../hooks/useLang';
import {useFlags} from '../../hooks/useFlags';

import CalendarDayLink from '../CalendarDayLink';

import threadKeyset from '../../i18n/thread';

interface IThreadCalendarDayLinkProps {
    day: Moment;
    today: Moment;
    context: ISearchContext;
    segment: ISegment;

    onClick?: React.MouseEventHandler;
}

export const ThreadCalendarDayLink: React.FC<IThreadCalendarDayLinkProps> =
    props => {
        const tld = useTld();
        const flags = useFlags();
        const language = useLang();
        const {day, today, context, segment, onClick} = props;

        let href;
        let linkTitle;

        if (dayMatchesMask(day, segment.runDays) && segment.departureMoment) {
            const localDepartureMoment = segment.departureMoment
                .clone()
                .tz(context.time.timezone)
                .set({
                    year: day.year(),
                    month: day.month(),
                    date: day.date(),
                });

            const threadUid = segment.isMerged
                ? getMaskValue(day, segment.runDays)
                : segment.thread && segment.thread.uid;
            const isCanonicalUrl = !flags?.__notCanonicalThreadUid;

            href = getThreadUrl({
                tld,
                language,
                threadUid,
                canonicalUid:
                    segment?.thread?.canonicalUid && isCanonicalUrl
                        ? segment.thread.canonicalUid
                        : undefined,
                departureFromMoment: localDepartureMoment,
                stationFromId: segment.stationFrom.id,
                stationToId: segment.stationTo.id,
                toCity: isCity(context.to),
            });

            linkTitle = threadKeyset(`title-${segment.transport.code}`, {
                number: segment.number,
                title: segment.title,
            });
        }

        return (
            <CalendarDayLink
                day={day}
                today={today}
                href={href}
                title={linkTitle}
                onClick={onClick}
                colorless={day.isBefore(today, 'day')}
            />
        );
    };

export default ThreadCalendarDayLink;
