import {React, PureComponent, B, PropTypes, mix} from '../../base';

import IconGlyph from '../../../interfaces/components/IconGlyph';

import noop from '../../../lib/noop';
import {reachGoal} from '../../../lib/yaMetrika';

import BottomSheet from '../../basic/BottomSheet';
import Icon from '../../Icon/Icon';
import Link from '../../Link';

import commonKeyset from '../../../i18n/common';

const b = B('SheetCompany');

const prefixes = {
    email: 'mailto',
    phone: 'tel',
};

const textActions = {
    email: commonKeyset('write-email'),
    phone: commonKeyset('call'),
};

const getPhoneNumber = text => text.replace(/[^\d+]/g, '');

export default class SheetCompany extends PureComponent {
    static propTypes = {
        isOpened: PropTypes.bool.isRequired,
        title: PropTypes.string.isRequired,
        transportType: PropTypes.string.isRequired,
        onClose: PropTypes.func.isRequired,

        address: PropTypes.string,
        url: PropTypes.string,
        description: PropTypes.string,
        email: PropTypes.string,
        phones: PropTypes.array,
        onClickOutside: PropTypes.func,
        className: PropTypes.string,
    };

    static defaultProps = {
        address: '',
        url: '',
        description: '',
        email: '',
        phones: [],
        onClickOutside: noop,
        className: '',
    };

    onClickPhone = () => {
        const {transportType} = this.props;

        reachGoal(`thread_${transportType}_carrier_phone_click`);
    };

    onClickWebsite = () => {
        const {transportType} = this.props;

        reachGoal(`thread_${transportType}_carrier_website_click`);
    };

    renderListItem(type, text) {
        const glyph =
            type === 'email'
                ? IconGlyph.email
                : type === 'phone'
                ? IconGlyph.handset
                : undefined;
        const link = prefixes[type]
            ? `${prefixes[type]}: ${
                  type === 'phone' ? getPhoneNumber(text) : text
              }`
            : '';
        const textAction = textActions[type] || '';
        let onClick;

        switch (type) {
            case 'phone':
                onClick = this.onClickPhone;
                break;
            case 'email':
                onClick = this.onClickWebsite;
                break;

            default:
                onClick = noop;
        }

        return (
            <a
                className={b('listItem')}
                href={link}
                key={link}
                onClick={onClick}
            >
                <span className={b('iconContainer')}>
                    {Boolean(glyph) && (
                        <Icon className={b('icon')} glyph={glyph} />
                    )}
                </span>
                <span className={b('actionContainer')}>{textAction}</span>
                <span className={b('textContainer')}>{text}</span>
            </a>
        );
    }

    render() {
        const {
            isOpened,
            onClose,
            title,
            address,
            url,
            description,
            phones,
            email,
            onClickOutside,
            className,
        } = this.props;

        return (
            <div className={mix(b(), className)}>
                <BottomSheet
                    opened={isOpened}
                    onClose={onClose}
                    onClickOutside={onClickOutside}
                >
                    <div className={b('title')}>{title}</div>
                    <div className={b('address')}>{address}</div>
                    {Boolean(url) && (
                        <Link href={url} target="_blank" rel="nofollow">
                            {url}
                        </Link>
                    )}
                    <div className={b('list')}>
                        {phones.map(phoneText =>
                            this.renderListItem('phone', phoneText),
                        )}
                        {Boolean(email) && this.renderListItem('email', email)}
                    </div>
                    <div className={b('description')}>{description}</div>
                </BottomSheet>
            </div>
        );
    }
}
