import {React, B} from '../../base';

import {useContext, useEffect, useRef, useCallback} from 'react';
import moment from 'moment';

import {TIME} from '../../../lib/date/formats';

import {EDurationShortening} from '../../../interfaces/date/EDurationShortening';
import IStation from '../../../interfaces/state/thread/IStation';
import ISegment from '../../../interfaces/segment/ISegment';
import IPrice from '../../../interfaces/state/IPrice';

// eslint-disable-next-line no-duplicate-imports
import {humanizeDuration} from '../../../lib/date/formats';
import formatPrice from '../../../lib/format/formatPrice';

import {ThreadLayoutContext} from '../../layout/ThreadLayout/ThreadLayoutContext';

import threadKeyset from '../../../i18n/thread';

const b = B('ThreadBottomDescription');
const isMobile = process.env.PLATFORM === 'mobile';

interface IDescriptionTextParams {
    number: string;
    title: string;
    stationFrom: string;
    stationTo: string;
    departureFrom: string;
    arrivalTo: string;
    stopsDescription: string;
    totalTime: string;
    technicalStopDescription: string;
}

function getTimeForDescription(time: number): string {
    return humanizeDuration(moment.duration(time), {
        short: EDurationShortening.ALWAYS,
    });
}

function getStopsDescription(stations: IStation[]): string {
    const stopsCount = stations.length - 2;

    if (stopsCount < 1) {
        return '';
    }

    const stationsSortedByTransitionTime = stations
        .slice(1, -1)
        .reduce((result, station) => {
            switch (true) {
                case !result.length:
                    return [station];
                case station.transitionTime > result[0].transitionTime:
                    return [station, result[0]];
                case !result[1] ||
                    station.transitionTime > result[1].transitionTime:
                    return [result[0], station];
            }

            return result;
        }, [] as IStation[]);

    const stopKeysetParams = {
        longestStation1Name: stationsSortedByTransitionTime[0].title,
        longestStation1Time: getTimeForDescription(
            stationsSortedByTransitionTime[0].transitionTime,
        ),
    };

    switch (true) {
        case stopsCount === 1:
            return threadKeyset('description-one-stop', stopKeysetParams);
        case stopsCount > 1:
            return threadKeyset('description-stops', {
                ...stopKeysetParams,
                longestStation2Name: stationsSortedByTransitionTime[1].title,
                longestStation2Time: getTimeForDescription(
                    stationsSortedByTransitionTime[1].transitionTime,
                ),
                stopsCount,
            });
    }

    return '';
}

function getDescriptionTextParams(
    shortTitle: string,
    stations: IStation[],
    number: string,
): IDescriptionTextParams | null {
    if (!stations || !stations.length) {
        return null;
    }

    const firstStation = stations[0];
    const lastStation = stations[stations.length - 1];

    const totalTimeDuration = moment(lastStation.arrivalLocalDt).diff(
        moment(firstStation.departureLocalDt),
    );

    const technicalStopDescription = stations.some(
        station => station.isTechnicalStop,
    )
        ? threadKeyset('description-technical-stop-stations')
        : '';

    const stopsDescription = getStopsDescription(stations);

    return {
        number,
        title: shortTitle,
        stationFrom: firstStation.title,
        stationTo: lastStation.title,
        departureFrom: moment
            .parseZone(firstStation.departureLocalDt)
            .format(TIME),
        arrivalTo: moment.parseZone(lastStation.arrivalLocalDt).format(TIME),
        stopsDescription,
        totalTime: getTimeForDescription(totalTimeDuration),
        technicalStopDescription,
    };
}

function getFormattedPriceAndType(price: IPrice, tariffKey: string): string {
    return `${formatPrice(price, {
        from: true,
    })} ${threadKeyset.get(`class-${tariffKey}`)}`;
}

function getDescriptionText(
    shortTitle: string,
    number: string,
    stations: IStation[],

    segment?: ISegment,
): string {
    const params = getDescriptionTextParams(shortTitle, stations, number);

    if (!params) {
        return '';
    }

    if (!segment) {
        return threadKeyset('description-text', {...params});
    }

    const {tariffClassKeys, tariffs} = segment;

    const classes = tariffs ? tariffs.classes : undefined;

    if (!tariffClassKeys || !tariffClassKeys.length || !classes) {
        return threadKeyset('description-text', {...params});
    }

    const tariffClassKeysWithSeats = tariffClassKeys.filter(
        className => classes[className] && classes[className].seats !== 0,
    );

    if (!tariffClassKeysWithSeats.length) {
        return threadKeyset('description-text', {...params});
    }

    if (tariffClassKeysWithSeats.length === 1) {
        const tariffKey = tariffClassKeysWithSeats[0];
        const {nationalPrice} = classes[tariffKey];

        return threadKeyset('description-text-with-prices', {
            ...params,
            pricesToTypes: getFormattedPriceAndType(nationalPrice, tariffKey),
        });
    }

    const pricesByClass = tariffClassKeysWithSeats.map(key => {
        const {nationalPrice} = classes[key];

        return getFormattedPriceAndType(nationalPrice, key);
    });

    const pricesToTypesWithoutLast = pricesByClass.slice(0, -1).join(', ');

    const pricesToTypes = `${pricesToTypesWithoutLast} ${threadKeyset(
        'price-in-type-last',
        {
            priceInType: pricesByClass[pricesByClass.length - 1],
        },
    )}`;

    return threadKeyset('description-text-with-prices', {
        ...params,
        pricesToTypes,
    });
}

interface IThreadBottomDescriptionProps {
    shortTitle: string;
    number: string;
    stations: IStation[];

    segment?: ISegment;
}

const ThreadBottomDescription = React.memo(
    ({
        segment,
        number,
        stations,
        shortTitle,
    }: IThreadBottomDescriptionProps): React.ReactElement => {
        const element = useRef<HTMLDivElement>(null);
        const {setMarginBottom} = useContext(ThreadLayoutContext);

        const setMarginBottomOfContainer = useCallback(() => {
            if (element.current) {
                const {offsetHeight} = element.current;

                setMarginBottom(offsetHeight);
            }
        }, [setMarginBottom]);

        useEffect(() => {
            window.addEventListener('resize', setMarginBottomOfContainer);

            return () => {
                window.removeEventListener(
                    'resize',
                    setMarginBottomOfContainer,
                );
            };
        }, [setMarginBottomOfContainer]);

        useEffect(() => {
            setMarginBottomOfContainer();
        });

        return (
            <div className={b({isMobile})} ref={element}>
                <h2 className={b('descriptionTitle')}>
                    {threadKeyset('description-title', {
                        number,
                        title: shortTitle,
                    })}
                </h2>
                <p className={b('descriptionText')}>
                    {getDescriptionText(shortTitle, number, stations, segment)}
                </p>
            </div>
        );
    },
);

export default ThreadBottomDescription;
