import {React, PureComponent, PropTypes, B} from '../../base';

import moment from 'moment';
import {connect} from 'react-redux';

import {ROBOT} from '../../../lib/date/formats';
import {PROP_TYPES, DEFAULT_PROPS} from '../propTypes';

import IconGlyph from '../../../interfaces/components/IconGlyph';

import getThreadUrl from '../../../lib/url/getThreadUrl';
import {getActivePlainMask, getMonthsByMask} from '../../../lib/date/mask';
import {
    getStationFrom,
    getStationTo,
    getFuncDataForDayComponent,
} from '../../../lib/thread/thread';
import {reachGoal} from '../../../lib/yaMetrika';

import {setCalendarIsOpened} from '../../../actions/thread';

import Icon from '../../Icon/Icon';
import Calendar from '../../Calendar/Calendar';
import ThreadPageCalendarDayLink from '../../ThreadPageCalendarDayLink';

const b = B('ThreadDates');

const isMobile = process.env.PLATFORM === 'mobile';

const mapStateToProps = ({language}) => ({language});

class ThreadDates extends PureComponent {
    static propTypes = {
        runDays: PROP_TYPES.runDays,
        stations: PROP_TYPES.stations,
        id: PROP_TYPES.id,
        canonicalUid: PROP_TYPES.canonicalUid,
        calendarIsOpened: PROP_TYPES.calendarIsOpened,
        isSetStationTo: PropTypes.bool.isRequired,
        transportType: PROP_TYPES.transportType,

        skipKeys: PropTypes.array,
        countDatesInline: PropTypes.number,
        currentTimezone: PROP_TYPES.currentTimezone,
        isToCitySearchContext: PropTypes.bool,
    };

    static defaultProps = {
        skipKeys: [],
        countDatesInline: 5,
        currentTimezone: DEFAULT_PROPS.currentTimezone,
    };

    onClickCalendarButton = () => {
        const {dispatch} = this.props;

        return dispatch(setCalendarIsOpened(true));
    };

    onCloseCalendar = () => {
        const {dispatch} = this.props;

        return dispatch(setCalendarIsOpened(false));
    };

    onClickDate = () => {
        const {transportType} = this.props;

        reachGoal(`thread_${transportType}_dates_click`);
    };

    render() {
        const {
            id,
            canonicalUid,
            countDatesInline,
            runDays,
            skipKeys,
            stations,
            calendarIsOpened,
            isSetStationTo,
            currentTimezone,
            language,
            tld,
            isToCitySearchContext,
        } = this.props;

        // определяем время станции отправления
        const stationFromData = getStationFrom({stations});
        const stationToData = getStationTo({
            stations,
            stationFrom: stationFromData,
        });

        if (!stationFromData) {
            throw new Error(
                `Не удалось определить станцию отправления. Data: ${JSON.stringify(
                    this.props,
                )}`,
            );
        }

        if (!stationToData) {
            throw new Error(
                `Не удалось определить станцию прибытия. Data: ${JSON.stringify(
                    this.props,
                )}`,
            );
        }

        const dateStart = moment.parseZone(stationFromData.departureLocalDt);
        const utcOffset = dateStart.utcOffset();
        const now = moment().utcOffset(utcOffset);

        const dayOfStart = dateStart.clone().startOf('day');

        const datesInCalendar = getActivePlainMask({
            mask: runDays,
            utcOffset,
            skipKeys,
        });
        const datesInLine = datesInCalendar
            .filter(date => date >= dayOfStart)
            .slice(0, countDatesInline);
        const months = getMonthsByMask({mask: runDays, utcOffset});

        return (
            <div className={b({isMobile})}>
                <div className={b('list')}>
                    {datesInLine.map((date, index) => (
                        <a
                            key={index}
                            className={b('item', {active: index === 0})}
                            onClick={this.onClickDate}
                            href={
                                index !== 0
                                    ? getThreadUrl({
                                          tld,
                                          language,
                                          departureFromMoment: date,
                                          stationFromId: stationFromData.id,
                                          stationToId: isSetStationTo
                                              ? stationToData.id
                                              : null,
                                          threadUid: id,
                                          canonicalUid,
                                          departureFromFormat: ROBOT,
                                          timezone: currentTimezone,
                                          toCity: isToCitySearchContext,
                                      })
                                    : null
                            }
                        >
                            <span className={b('itemDate')} data-nosnippet>
                                {date.format('D MMM')}
                            </span>
                            <span className={b('itemWeekday')}>
                                {date.format('dd').toUpperCase()}
                            </span>
                        </a>
                    ))}
                </div>

                <div className={b('openCalendarContainer')}>
                    {isMobile && (
                        <div
                            className={b('calendarButtonWrapper')}
                            onClick={this.onClickCalendarButton}
                        >
                            <Icon
                                className={b('calendarIcon')}
                                glyph={IconGlyph.calendarSmall}
                            />
                        </div>
                    )}
                </div>

                {isMobile && (
                    <Calendar
                        id="thread"
                        opened={calendarIsOpened}
                        time={now}
                        value={{date: dateStart.format(ROBOT)}}
                        language={language}
                        showToolbar={false}
                        onClose={this.onCloseCalendar}
                        onClick={this.onClickDate}
                        months={months}
                        DayComponent={ThreadPageCalendarDayLink}
                        getDataForDayComponent={getFuncDataForDayComponent({
                            tld,
                            language,
                            stationFromId: stationFromData.id,
                            stationToId: isSetStationTo
                                ? stationToData.id
                                : null,
                            datesInCalendar,
                            runDays,
                            threadId: id,
                            canonicalUid,
                            stations,
                            timezone: currentTimezone,
                            isToCitySearchContext,
                        })}
                    />
                )}
            </div>
        );
    }
}

export default connect(mapStateToProps)(ThreadDates);
