import {React, PureComponent, PropTypes, B, mix} from '../../base';

import moment from 'moment';
import {connect} from 'react-redux';

import {PROP_TYPES, DEFAULT_PROPS} from '../propTypes';

import IconGlyph from '../../../interfaces/components/IconGlyph';

import {reachGoal, reachGoalOnce} from '../../../lib/yaMetrika';
import {setMapIsOpened} from '../../../actions/thread';

import Duration from '../../Duration/Duration';
import Icon from '../../Icon/Icon';
import ThreadTimezonesSwitch from '../ThreadTimezonesSwitch';

import timezonesKeyset from '../../../i18n/timezones';
import threadKeyset from '../../../i18n/thread';

const b = B('ThreadInfo');

class ThreadInfo extends PureComponent {
    static propTypes = {
        id: PROP_TYPES.id,
        canonicalUid: PROP_TYPES.canonicalUid,
        capitals: PROP_TYPES.capitals,
        stationFrom: PROP_TYPES.stationFrom,
        stationTo: PROP_TYPES.stationTo,
        departure: PROP_TYPES.departure,
        departureFrom: PROP_TYPES.departureFrom,
        capitalSlug: PROP_TYPES.capitalSlug,
        transportType: PROP_TYPES.transportType,
        fromStationDepartureLocalDt: PROP_TYPES.fromStationDepartureLocalDt,
        dateStart: PropTypes.instanceOf(moment).isRequired,
        dateStop: PropTypes.instanceOf(moment).isRequired,

        currentTimezone: PROP_TYPES.currentTimezone,
        shouldShowTimezoneSwitch: PropTypes.bool,
        mapData: PropTypes.object,
        className: PropTypes.string,
    };

    static defaultProps = {
        currentTimezone: DEFAULT_PROPS.currentTimezone,
        shouldShowTimezoneSwitch: false,
        mapData: DEFAULT_PROPS.mapData,
        className: '',
    };

    componentDidMount() {
        const {mapData, transportType} = this.props;

        if (mapData) {
            reachGoalOnce(`thread_${transportType}_map_show`);
        }
    }

    onClickMapButton = () => {
        const {transportType, dispatch} = this.props;

        reachGoal(`thread_${transportType}_map_click`);

        return dispatch(setMapIsOpened(true));
    };

    onClickPrint = () => {
        const {transportType} = this.props;

        reachGoal(`thread_${transportType}_print_click`);
        window.print();
    };

    renderLocalTime() {
        const {dateStart, dateStop} = this.props;

        return (
            <div className={b('info')}>
                <span className={b('localTime')}>
                    {timezonesKeyset('local-time-specified')}
                </span>

                <span className={b('inTheWay')}>
                    <span className={b('inTheWayText')}>
                        {threadKeyset('in-the-way')}&nbsp;
                    </span>
                    <Duration duration={dateStop - dateStart} />
                </span>
            </div>
        );
    }

    renderTimezoneSwitch() {
        const {
            id,
            canonicalUid,
            capitals,
            currentTimezone,
            stationFrom,
            stationTo,
            departure,
            departureFrom,
            fromStationDepartureLocalDt,
        } = this.props;

        return (
            <div className={b('info')}>
                <ThreadTimezonesSwitch
                    id={id}
                    canonicalUid={canonicalUid}
                    capitals={capitals}
                    stationFrom={stationFrom}
                    stationTo={stationTo}
                    departure={departure}
                    departureFrom={departureFrom}
                    currentTimezone={currentTimezone}
                    fromStationDepartureLocalDt={fromStationDepartureLocalDt}
                />
            </div>
        );
    }

    render() {
        const {className, mapData, shouldShowTimezoneSwitch} = this.props;

        return (
            <div className={mix(b(), className)}>
                {shouldShowTimezoneSwitch
                    ? this.renderTimezoneSwitch()
                    : this.renderLocalTime()}

                {mapData && (
                    <div
                        className={b('mapButton')}
                        onClick={this.onClickMapButton}
                    >
                        <Icon
                            className={b('mapIcon')}
                            glyph={IconGlyph.mapIcon}
                        />

                        <span className={b('mapText')}>
                            {threadKeyset('map-button')}
                        </span>
                    </div>
                )}

                <div className={b('printButton')} onClick={this.onClickPrint}>
                    <Icon
                        className={b('printIcon')}
                        glyph={IconGlyph.printIcon}
                    />

                    <span className={b('printText')}>
                        {threadKeyset('print-the-route')}
                    </span>
                </div>
            </div>
        );
    }
}

export default connect()(ThreadInfo);
