import {PureComponent} from 'react';

import {SUBURBAN_TYPE, TRAIN_TYPE} from '../../lib/transportType';

import Platform from '../../interfaces/Platform';

import patchTitle from '../../lib/segments/patchTitle';
import makeUrl from '../../lib/url/makeUrl';
import {reachGoalOnce} from '../../lib/yaMetrika';

const isMobile = process.env.PLATFORM === Platform.mobile;

export default class ThreadPageBase extends PureComponent {
    componentDidMount() {
        const {page, thread, tld, language} = this.props;
        const {company, transportType} = thread;

        if (!page.fetching) {
            this.reachGoalThreadPage({
                transportType,
                company,
                phones: this.phones,
            });
        }

        if (isMobile) {
            window.addEventListener('popstate', this.popstateHandler);

            if (page.prevPage === 'search') {
                const url = makeUrl(
                    page.location.pathname,
                    tld,
                    language,
                    page.location.query,
                );

                history.replaceState({path: url}, '', url);
                window.removeEventListener('popstate', this.popstateHandler);
            }
        }
    }

    componentDidUpdate(prevProps) {
        const {page: prevPage} = prevProps;
        const {page, thread} = this.props;
        const {company, transportType} = thread;

        if (prevPage.fetching && !page.fetching) {
            this.reachGoalThreadPage({
                transportType,
                company,
                phones: this.phones,
            });
        }
    }

    componentWillUnmount() {
        window.removeEventListener('popstate', this.popstateHandler);
    }

    popstateHandler = () => {
        if (isMobile) {
            history.forward();
        }
    };

    reachGoalThreadPage({transportType, company, phones}) {
        reachGoalOnce(`thread_${transportType}_show`);

        if (isMobile) {
            const {title, hidden} = company;

            if (title && !hidden) {
                reachGoalOnce(`thread_${transportType}_carrier_show`);
            }

            if (this.shouldShowCompanyInfo()) {
                reachGoalOnce(`thread_${transportType}_carrier_contact_show`);
            }
        } else {
            this.reachGoalCompanyInfo({company, phones, transportType});
        }
    }

    reachGoalCompanyInfo({company, phones, transportType}) {
        const {strange, url, email} = company;

        if (!isMobile) {
            reachGoalOnce(`thread_${transportType}_carrier_show`);
        }

        if (strange) {
            return;
        }

        if (phones.length) {
            reachGoalOnce(`thread_${transportType}_carrier_phone_show`);
        }

        if (url) {
            reachGoalOnce(`thread_${transportType}_carrier_website_show`);
        }

        if (email) {
            reachGoalOnce(`thread_${transportType}_carrier_email_show`);
        }
    }

    getPhoneNumber(phoneText) {
        return phoneText.replace(/[^\d+]/g, '');
    }

    get title() {
        const {title} = this.props.thread;

        return patchTitle(title);
    }

    get phones() {
        const {phone} = this.props.thread.company;

        return phone
            .split(',')
            .map(text => text.trim())
            .filter(text => text && this.getPhoneNumber(text));
    }

    shouldShowCompanyInfo() {
        const {company} = this.props.thread;

        const {email, phone, address, description, url, strange} = company;
        const existsInfoFields =
            email || phone || address || description || url;

        return (
            Boolean(company.title) &&
            !company.hidden &&
            (!isMobile || (existsInfoFields && !strange))
        );
    }

    shouldShowTimezoneSwitch() {
        const {capitals, transportType} = this.props.thread;

        return Boolean(
            (transportType === SUBURBAN_TYPE || transportType === TRAIN_TYPE) &&
                capitals.length,
        );
    }
}
