import React, {PureComponent} from 'react';
import {connect} from 'react-redux';
import B from 'bem-cn-lite';

import {THREAD_PAGE_NAME} from '../../../routes/thread';

import {TransportType} from '../../../lib/transportType';
import IStateCurrencies from '../../../interfaces/state/IStateCurrencies';
import IStateFlags from '../../../interfaces/state/flags/IStateFlags';
import IStateThread from '../../../interfaces/state/stateThread/IStateThread';
import IStateSeoQueryParams from '../../../interfaces/state/IStateSeoQueryParams';
import IStation from '../../../interfaces/state/thread/IStation';
import MapStateToProps from '../../../interfaces/MapStateToProps';

import getTariffKeys from '../../../lib/segments/getTariffKeys';
import {getBuyLinkText} from '../../../lib/segments/snippetUtils';
import getLowestPrice from '../../../lib/segments/getLowestPrice';
import {reachGoal, reachGoalOnce} from '../../../lib/yaMetrika';
import {getStationFrom, getStationTo} from '../../../lib/thread/thread';
import convertPrice from '../../../lib/currencies/convertPrice';

import Price from '../../Price/Price';
import Spinner from '../../Spinner/Spinner';
import Button2 from '../../Button2/Button2';
import SegmentBuyLink from '../../SegmentBuyLink';
import TariffsList from '../../TariffsList/TariffsList';

import threadKeyset from '../../../i18n/thread';

const b = B('ThreadPrices');

const mapStateToProps: MapStateToProps<IThreadPricesProps> = ({
    thread,
    currencies,
    seoQueryParams,
    flags,
}) => ({
    thread,
    currencies,
    seoQueryParams,
    flags,
});

interface IThreadPricesProps {
    thread: IStateThread;
    currencies: IStateCurrencies;
    seoQueryParams: IStateSeoQueryParams;
    flags: IStateFlags;

    className?: string;
}

class ThreadPrices extends PureComponent<IThreadPricesProps, {}> {
    onShowBuyButton = (): void => {
        const {transportType} = this.props.thread;

        reachGoalOnce(`thread_${transportType}_price_button_show`);
    };

    onClickBuyButton = (): void => {
        const {transportType} = this.props.thread;

        reachGoal(`thread_${transportType}_price_button_click`);
    };

    render(): React.ReactElement | null {
        const {className, thread, currencies, flags, seoQueryParams} =
            this.props;
        const {segment, transportType, priceQuerying, stations} = thread;

        if (priceQuerying) {
            return (
                <div className={b('buy', {priceQuerying})}>
                    <Spinner size="large" />
                </div>
            );
        }

        if (!segment) {
            return null;
        }

        const cheapestPrice = getLowestPrice(segment);

        if (!cheapestPrice) {
            return null;
        }

        const {mainTariffKey, tariffKeys} = getTariffKeys(
            segment,
            flags,
            seoQueryParams,
        );

        if (Boolean(mainTariffKey) === false) {
            return null;
        }

        if (
            transportType !== TransportType.train &&
            transportType !== TransportType.bus &&
            transportType !== TransportType.suburban
        ) {
            return null;
        }

        const stationFromData: IStation = getStationFrom({
            stations: stations || [],
        });
        const stationToData: IStation = getStationTo({
            stations: stations || [],
            stationFrom: stationFromData,
        });

        return (
            <div className={b({}, className)}>
                {(transportType === TransportType.train ||
                    transportType === TransportType.suburban) && (
                    <div className={b('trainOfferInfo')}>
                        <span className={b('trainOfferTitle')}>
                            {threadKeyset.get('tickets-for-from-to', {
                                titleFrom: stationFromData.title,
                                titleTo: stationToData.title,
                            })}
                        </span>

                        <div>
                            <TariffsList
                                segment={segment}
                                tariffClassKeys={tariffKeys}
                                currencies={currencies}
                                viewMode="detailed"
                            />
                        </div>
                    </div>
                )}

                {transportType === TransportType.bus && (
                    <div className={b('busOfferInfo')}>
                        <span className={b('busOfferTitle')}>
                            {threadKeyset.get('tickets-for-from-to', {
                                titleFrom: stationFromData.title,
                                titleTo: stationToData.title,
                            })}
                        </span>

                        <Price
                            className={b('busPrice')}
                            from={transportType !== TransportType.bus}
                            price={convertPrice(cheapestPrice, currencies)}
                        />
                    </div>
                )}

                <SegmentBuyLink
                    className={b('buyLink')}
                    segment={segment}
                    tariffClass={mainTariffKey}
                    pageType={THREAD_PAGE_NAME}
                    onClick={this.onClickBuyButton}
                    onShow={this.onShowBuyButton}
                >
                    <Button2
                        className={b('priceAction')}
                        type="button"
                        theme="action"
                    >
                        {transportType === TransportType.train
                            ? threadKeyset.get('select-and-buy-ticket')
                            : getBuyLinkText(segment)}
                    </Button2>
                </SegmentBuyLink>
            </div>
        );
    }
}

export default connect(mapStateToProps)(ThreadPrices);
