import React, {memo, useCallback, useMemo} from 'react';
import B from 'bem-cn-lite';

import {THREAD_PAGE_NAME} from '../../../routes/thread';

import {TransportType} from '../../../lib/transportType';

import getTariffKeys from '../../../lib/segments/getTariffKeys';
import {getBuyLinkText} from '../../../lib/segments/snippetUtils';
import getLowestPrice from '../../../lib/segments/getLowestPrice';
import {reachGoal, reachGoalOnce} from '../../../lib/yaMetrika';
import convertPrice from '../../../lib/currencies/convertPrice';
import insertMarkupIntoKey from '../../../i18n/_utils/insertMarkupIntoKey';
import {isIOS} from '../../../lib/os';
import useVisibility from '../../../lib/useVisibility';

import useSelector from '../../useSelector';
import {useFlags} from '../../../hooks/useFlags';

import TariffsList from '../../TariffsList/TariffsList';
import SegmentBuyLink from '../../SegmentBuyLink';
import Spinner from '../../Spinner/Spinner';
import Price from '../../Price/Price';
import ButtonSkeleton from '../../ButtonSkeleton/ButtonSkeleton';
import TextSkeleton from '../../TextSkeleton/TextSkeleton';

import threadKeyset from '../../../i18n/thread';
import priceKeyset from '../../../i18n/price';

const b = B('ThreadPrices');

const BUY_BUTTON_SIZE = 44;

interface IThreadPricesProps {
    className?: string;
}

export default memo(ThreadPrices);

function ThreadPrices({
    className,
}: IThreadPricesProps): React.ReactElement | null {
    const flags = useFlags();
    const thread = useSelector(state => state.thread);
    const currencies = useSelector(state => state.currencies);
    const seoQueryParams = useSelector(state => state.seoQueryParams);
    const user = useSelector(state => state.user);

    const isIOSPlatform = isIOS(user.os);

    const [pricesIsVisible, buttonRef] = useVisibility(BUY_BUTTON_SIZE);

    const {transportType, segment, priceQuerying} = thread;

    const onShowBuyButton = useCallback(() => {
        reachGoalOnce(`thread_${transportType}_price_button_show`);
    }, [transportType]);

    const onShowStaticPrice = useCallback(() => {
        reachGoalOnce(`thread_${transportType}_price_show`);
    }, [transportType]);

    const onClickBuyButton = useCallback(() => {
        reachGoal(`thread_${transportType}_price_button_click`);
    }, [transportType]);

    const {mainTariffKey, tariffKeys} = getTariffKeys(
        segment,
        flags,
        seoQueryParams,
    );

    const prices = useMemo(() => {
        if (!segment) {
            return null;
        }

        switch (transportType) {
            case TransportType.suburban:
            case TransportType.train:
                return (
                    <div className={b('scrollableWrapper')}>
                        <div className={b('scrollable')}>
                            <div className={b('scrollableContent')}>
                                <TariffsList
                                    segment={segment}
                                    tariffClassKeys={tariffKeys}
                                    currencies={currencies}
                                    pageType={THREAD_PAGE_NAME}
                                />
                            </div>
                        </div>
                    </div>
                );
            case TransportType.bus: {
                const cheapestPrice = getLowestPrice(segment);

                if (!cheapestPrice) {
                    return null;
                }

                return (
                    <div className={b('busPriceContainer')}>
                        {insertMarkupIntoKey(priceKeyset, 'from', {
                            price: (
                                <Price
                                    className={b('busPrice')}
                                    key="price"
                                    onShow={
                                        mainTariffKey
                                            ? undefined
                                            : onShowStaticPrice
                                    }
                                    price={convertPrice(
                                        cheapestPrice,
                                        currencies,
                                    )}
                                />
                            ),
                        })}
                    </div>
                );
            }
        }
    }, [
        currencies,
        transportType,
        segment,
        mainTariffKey,
        tariffKeys,
        onShowStaticPrice,
    ]);

    const segmentBuyLink = useMemo(() => {
        if (!segment || !mainTariffKey) {
            return null;
        }

        return (
            <SegmentBuyLink
                className={b('buyLink')}
                segment={segment}
                tariffClass={mainTariffKey}
                pageType={THREAD_PAGE_NAME}
                onClick={onClickBuyButton}
                onShow={onShowBuyButton}
            >
                {getBuyLinkText(segment)}
            </SegmentBuyLink>
        );
    }, [segment, mainTariffKey, onClickBuyButton, onShowBuyButton]);

    const buyContent = useMemo(() => {
        if (priceQuerying) {
            switch (transportType) {
                case TransportType.bus: {
                    return (
                        <div className={b('busSkeleton')}>
                            <TextSkeleton
                                className={b('busFirstTextSkeleton')}
                                width={64}
                                size="m"
                            />
                            <ButtonSkeleton />
                        </div>
                    );
                }

                case TransportType.suburban:
                case TransportType.train: {
                    return (
                        <div className={b('trainSkeleton')}>
                            <div className={b('trainPricesSkeleton')}>
                                <div className={b('trainPriceSkeleton')}>
                                    <TextSkeleton width={57} />
                                    <TextSkeleton width={64} size="m" />
                                </div>

                                <div className={b('trainPriceSkeleton')}>
                                    <TextSkeleton width={57} />
                                    <TextSkeleton width={64} size="m" />
                                </div>

                                <div className={b('trainPriceSkeleton')}>
                                    <TextSkeleton width={57} />
                                    <TextSkeleton width={64} size="m" />
                                </div>
                            </div>
                            <ButtonSkeleton />
                        </div>
                    );
                }

                default:
                    return (
                        <div className={b('buy', {priceQuerying})}>
                            <Spinner size="large" />
                        </div>
                    );
            }
        }

        if (!segment) {
            return null;
        }

        return (
            <div className={b('buy')}>
                {prices}

                <div ref={buttonRef}>{segmentBuyLink}</div>
            </div>
        );
    }, [
        segment,
        priceQuerying,
        prices,
        transportType,
        buttonRef,
        segmentBuyLink,
    ]);

    if (
        transportType !== TransportType.train &&
        transportType !== TransportType.bus &&
        transportType !== TransportType.suburban
    ) {
        return null;
    }

    const mods = {
        transportType,
        withoutBuyButton: segment && !mainTariffKey,
    };

    const title =
        transportType === TransportType.bus
            ? threadKeyset('title-prices-bus')
            : threadKeyset('title-prices');

    return (
        <div className={b(mods, className)}>
            {transportType === TransportType.bus && priceQuerying ? (
                <TextSkeleton
                    className={b('busTitleSkeleton')}
                    width={45}
                    withAnimation
                />
            ) : (
                <h3 className={b('title')}>{title}</h3>
            )}

            {buyContent}

            {!pricesIsVisible && (
                <div className={b('buyLinkBottomContainer', {isIOSPlatform})}>
                    {segmentBuyLink}
                </div>
            )}
        </div>
    );
}
