import {React, PropTypes, PureComponent, B} from '../../base';

import camelCase from 'lodash/camelCase';
import moment from 'moment';
import {connect} from 'react-redux';

import {CHAR_EM_DASH} from '../../../lib/stringUtils';
import {CANCEL, NON_CONTEXT} from '../../../lib/constants/threadScheduleTypes';
import {PROP_TYPES, DEFAULT_PROPS} from '../propTypes';

import getThreadUrl from '../../../lib/url/getThreadUrl';
import {reachGoal, reachGoalOnce} from '../../../lib/yaMetrika';

import Link from '../../Link';

import threadKeyset from '../../../i18n/thread';

const b = B('ThreadSchedule');

const mapStateToProps = ({tld, language}) => ({
    tld,
    language,
});

class ThreadSchedule extends PureComponent {
    static propTypes = {
        type: PropTypes.string.isRequired,
        transportType: PROP_TYPES.transportType,
        canonicalUid: PROP_TYPES.canonicalUid,
        threads: PropTypes.array.isRequired,
        stationFromId: PropTypes.number.isRequired,
        stationFromTitle: PropTypes.string.isRequired,
        stationToId: PropTypes.number.isRequired,
        stationToTitle: PropTypes.string.isRequired,

        currentTimezone: PROP_TYPES.currentTimezone,
    };

    static defaultProps = {
        currentTimezone: DEFAULT_PROPS.currentTimezone,
    };

    componentDidMount() {
        const {transportType} = this.props;

        reachGoalOnce(`thread_${transportType}_selector_show`);
    }

    onClickScheduleItem = () => {
        const {transportType} = this.props;

        reachGoal(`thread_${transportType}_selector_click`);
    };

    getScheduleList() {
        const {
            type,
            stationFromId,
            stationToId,
            canonicalUid,
            threads,
            currentTimezone,
            tld,
            language,
        } = this.props;

        return threads.map(thread => {
            const {
                title,
                inContext,
                firstDeparture,
                startDepartureTime,
                stopArrivalTime,
                daysText,
                key,
            } = thread;
            const threadUrlParams = inContext
                ? {
                      departureFromMoment: moment.parseZone(firstDeparture),
                      stationFromId,
                      stationToId,
                  }
                : {
                      departure: firstDeparture,
                  };

            return (
                <div className={b('scheduleItem')} key={key}>
                    {type !== CANCEL && (
                        <Link
                            className={b('threadLink')}
                            href={getThreadUrl({
                                tld,
                                language,
                                canonicalUid,
                                timezone: currentTimezone,
                                ...threadUrlParams,
                            })}
                            onClick={this.onClickScheduleItem}
                        >
                            {title}
                        </Link>
                    )}
                    {type !== CANCEL && (
                        <div className={b('times')}>
                            {`${startDepartureTime} ${CHAR_EM_DASH} ${stopArrivalTime}`}
                        </div>
                    )}
                    <p className={b('daysText')} data-nosnippet>
                        {daysText}
                    </p>
                </div>
            );
        });
    }

    render() {
        const {type, stationFromTitle, stationToTitle, threads} = this.props;
        const noColorTypes = [CANCEL, NON_CONTEXT];

        if (!threads.length) {
            return null;
        }

        return (
            <div className={b({type: camelCase(type)})}>
                <div className={b('title')}>
                    {!noColorTypes.includes(type) && (
                        <div className={b('scheduleColor')} />
                    )}
                    <span className={b('scheduleTitle')}>
                        {threadKeyset(`schedule-${type}`, {
                            stationFromTitle,
                            stationToTitle,
                        })}
                    </span>
                </div>
                <div className={b('schedulesList')}>
                    {this.getScheduleList()}
                </div>
            </div>
        );
    }
}

export default connect(mapStateToProps)(ThreadSchedule);
