import React, {memo, useCallback, useMemo} from 'react';
import B from 'bem-cn-lite';

import moment, {Moment} from 'moment';

import {CHAR_EM_DASH} from '../../../lib/stringUtils';
import {TLD_UA} from '../../../lib/tlds';
// eslint-disable-next-line no-duplicate-imports
import {formatTimeDifference, ROBOT} from '../../../lib/date/formats';

import Platform from '../../../interfaces/Platform';
import ICapital from '../../../interfaces/ICapital';
import IStation from '../../../interfaces/state/thread/IStation';
import {TransportType} from '../../../lib/transportType';
import {EDurationShortening} from '../../../interfaces/date/EDurationShortening';
import FactTypes from '../../../interfaces/state/stateThread/factInfo/FactTypes';
import IFactInfo from '../../../interfaces/state/stateThread/factInfo/IFactInfo';
import {
    IGetStationRowParams,
    IGetSubtitleRowParams,
    IGetTransitionRowParams,
    isStationTableRow,
    isStationTableRowParams,
    isSubtitleTableRow,
    isSubtitleTableRowParams,
    isTransitionTableRowParams,
    ISubtitleTableRow,
    TGetRowParams,
    ThreadTableColumnName,
    ThreadTableRowType,
    ThreadType,
    TTableRowContent,
} from './types';

import {
    setNextStationsIsOpened,
    setPrevStationsIsOpened,
} from '../../../actions/thread';
import {getFlagUrl} from '../../../lib/flagsUrl';
import {reachGoal, reachGoalOnce} from '../../../lib/yaMetrika';
import {stationUrl} from '../../../lib/url/stationUrl';
import getSubtypeByTransportType from '../../../lib/station/getSubtypeByTransportType';

import useSelector from '../../../components/useSelector';
import useDispatch from '../../../components/useDispatch';

import Link from '../../Link';
import Duration from '../../Duration/Duration';
import Arrow from '../../Arrow/Arrow';
import NoticeStar from '../../NoticeStar';
import ThreadTimezonesSwitch from '../ThreadTimezonesSwitch';

import threadKeyset from '../../../i18n/thread';
import timezoneKeyset from '../../../i18n/timezones';

const b = B('ThreadTable');

const isMobile = process.env.PLATFORM === Platform.mobile;
const formatArrDepDate = 'D MMM';

const DEFAULT_VALUE = CHAR_EM_DASH;

interface INeedDisplayColumnParams {
    columnName: ThreadTableColumnName;
    isIntervalThread: boolean;
    isSuburbanBus: boolean;
}

function needDisplayColumn({
    columnName,
    isIntervalThread,
    isSuburbanBus,
}: INeedDisplayColumnParams): boolean {
    switch (columnName) {
        case ThreadTableColumnName.station:
            return true;
        case ThreadTableColumnName.arrival:
            return !isIntervalThread && !isSuburbanBus;
        case ThreadTableColumnName.stay:
            return !isIntervalThread && !isSuburbanBus;
        case ThreadTableColumnName.departure:
            return !isIntervalThread;
        case ThreadTableColumnName.timeInTheWay:
            return !isMobile || isIntervalThread;
        case ThreadTableColumnName.intervalInfo:
            return isIntervalThread;
    }
}

interface IGetTimezoneTextParams {
    row: ISubtitleTableRow;
    capitals: ICapital[];
    capitalSlug: string;
}

function getTimezoneText({
    row,
    capitals,
    capitalSlug,
}: IGetTimezoneTextParams): string {
    const {capitalTimeOffset} = row;

    if (!capitalTimeOffset) {
        return '';
    }

    let capitalShortName = timezoneKeyset.get(`${capitalSlug}-short`);

    if (!capitalShortName) {
        const capital = capitals.find(({slug}) => slug === capitalSlug);

        if (!capital) {
            throw new Error(
                'Не удалось определить столицу для отображения ' +
                    'перехода через временную зону.',
            );
        }

        const {title, abbrTitle} = capital;

        capitalShortName = (abbrTitle && abbrTitle.toUpperCase()) || title;
    }

    const offsetInMinutes = moment().utcOffset(capitalTimeOffset).utcOffset();

    if (offsetInMinutes === 0 && timezoneKeyset.hasKey(capitalSlug)) {
        // Чтобы вместо "Местное время МСК +0 ч" писать "Московское время"
        return timezoneKeyset(capitalSlug);
    }

    const timezoneDifference = formatTimeDifference(offsetInMinutes);

    return threadKeyset(`timezone-change${isMobile ? '-short' : ''}`, {
        capitalShortName,
        timezoneDifference,
    });
}

function getSubtitle({
    row,
    index,
    capitals,
    capitalSlug,
}: IGetSubtitleRowParams): React.ReactElement {
    const {country, colspan, stationDate} = row;

    const timezoneText = getTimezoneText({row, capitals, capitalSlug});

    return (
        <tr className={b('rowSubtitle')} key={index}>
            <td colSpan={colspan}>
                <div className={b('wrapper')}>
                    <div className={b('wrapperInner')}>
                        <div className={b('subtitleContainer')}>
                            {stationDate && (
                                <span
                                    className={b('subtitleElement')}
                                    data-nosnippet
                                >
                                    {stationDate.format('D MMMM').toLowerCase()}
                                </span>
                            )}
                            {country && (
                                <span className={b('subtitleElement')}>
                                    <img
                                        className={b('countryFlag')}
                                        src={getFlagUrl(country.code)}
                                        alt={country.code}
                                    />
                                    {country.title}
                                </span>
                            )}
                            {timezoneText && (
                                <span className={b('subtitleElement')}>
                                    {timezoneText}
                                </span>
                            )}
                            <span className={b('subtitleLine')} />
                        </div>
                    </div>
                </div>
            </td>
        </tr>
    );
}

function getTransition({
    row,
    index,
}: IGetTransitionRowParams): React.ReactElement {
    const {transitionTime, colspan} = row;

    return (
        <tr className={b('rowTransition')} key={index}>
            <td colSpan={colspan}>
                <div className={b('wrapper')}>
                    <div className={b('wrapperInner')}>
                        <div className={b('transition')}>
                            {`${threadKeyset('transition')} `}
                            <Duration
                                duration={transitionTime}
                                options={{short: EDurationShortening.ALWAYS}}
                            />
                        </div>
                    </div>
                </div>
            </td>
        </tr>
    );
}

interface IFormatTime {
    time: string;
    isFuzzy: boolean;
}

function formatTime({time, isFuzzy}: IFormatTime): string {
    return time ? `${isFuzzy ? '~' : ''}${time}` : DEFAULT_VALUE;
}

interface IGetFactInfo {
    factInfo: IFactInfo;

    message?: string;
}

function getFactInfo({factInfo, message}: IGetFactInfo): React.ReactElement {
    const type =
        (factInfo.departure && factInfo.departure.type) ||
        (factInfo.arrival && factInfo.arrival.type) ||
        FactTypes.OK;

    return <div className={b('fact', {type})}>{message || ''}</div>;
}

function getStation({
    row,
    startDateOfThread,
    index,
    stationFromData,
    stationToData,
    beginTime,
    endTime,
    density,
    comment,
    transportType,
    isSuburbanBus,
    isIntervalThread,
    tld,
    language,
    onClickRowStation,
    onClickStationName,
    stations,

    displayIntervalThreadInfo = false,
    isFullCanceling,
}: IGetStationRowParams): React.ReactElement {
    const {station, thread, departureDate, arrivalDate} = row;
    const {
        title,
        id: stationId,
        settlement,
        platform,
        mainSubtype,
        pageType,
        factInfo,
        isFuzzy,
        isNoStop,
        isTechnicalStop,
    } = station;

    const isStationFrom =
        thread === ThreadType.current && station === stationFromData;
    const isStationTo =
        thread === ThreadType.current && station === stationToData;
    const arrival = arrivalDate ? arrivalDate.format('HH:mm') : '';
    const departure = departureDate ? departureDate.format('HH:mm') : '';

    const arrivalAndDepartureIsDifferentDay = Boolean(
        arrivalDate &&
            departureDate &&
            !arrivalDate.isSame(departureDate, 'day'),
    );
    const stationLink = stationUrl({
        id: stationId,
        subtype: getSubtypeByTransportType(transportType),
        type: pageType,
        mainSubtype,
        isMobile,
        tld,
        language,
    });

    const stationFactInfoTypes = [
        factInfo?.arrival?.type,
        factInfo?.departure?.type,
    ];
    const allStationDirectionsCanceled = stationFactInfoTypes.every(
        currType => currType === FactTypes.CANCELLED,
    );
    const anyStationDirectionsCanceled = stationFactInfoTypes.some(
        currType => currType === FactTypes.CANCELLED,
    );
    const onlyOneStationDirectionCanceled =
        !allStationDirectionsCanceled && anyStationDirectionsCanceled;
    const isIntermediateStation = !isStationFrom && !isStationTo;
    const isNoStopsAndBePartOfCanceling =
        isNoStop &&
        ([
            stations
                .slice(0, index)
                .some(prevStation =>
                    [
                        prevStation.factInfo?.arrival?.type,
                        prevStation.factInfo?.departure?.type,
                    ].some(currType => currType === FactTypes.CANCELLED),
                ),
            stations
                .slice(index + 1)
                .some(nextStation =>
                    [
                        nextStation.factInfo?.arrival?.type,
                        nextStation.factInfo?.departure?.type,
                    ].some(currType => currType === FactTypes.CANCELLED),
                ),
        ].every(value => value) ||
            isFullCanceling);

    return (
        <tr
            className={b('rowStation', {isStationFrom, isStationTo})}
            key={index}
            data-href={stationLink}
            onClick={onClickRowStation}
        >
            {needDisplayColumn({
                columnName: ThreadTableColumnName.station,
                isIntervalThread,
                isSuburbanBus,
            }) && (
                <td className={b('station')}>
                    <div className={b('wrapper')}>
                        <div className={b('wrapperInner')}>
                            <Link
                                href={stationLink}
                                onClick={onClickStationName}
                                className={b('stationLink')}
                            >
                                {title}
                                {isTechnicalStop && <NoticeStar />}
                            </Link>
                            {Boolean(platform) && (
                                <div className={b('stationInfo')}>
                                    {platform}
                                </div>
                            )}
                            {Boolean(settlement.title) && isSuburbanBus && (
                                <div className={b('stationSettlement')}>
                                    {settlement.title}
                                </div>
                            )}
                            {factInfo &&
                                Boolean(factInfo) &&
                                !isNoStop &&
                                getFactInfo({
                                    factInfo,
                                    message:
                                        onlyOneStationDirectionCanceled &&
                                        isIntermediateStation
                                            ? ''
                                            : factInfo.message,
                                })}

                            {isNoStopsAndBePartOfCanceling && (
                                <div className={b('fact', {type: 'cancelled'})}>
                                    {threadKeyset('cancelled')}
                                </div>
                            )}
                        </div>
                    </div>
                </td>
            )}
            {needDisplayColumn({
                columnName: ThreadTableColumnName.arrival,
                isSuburbanBus,
                isIntervalThread,
            }) && (
                <td className={b('arrival')}>
                    <div className={b('wrapper')}>
                        <div className={b('wrapperInner')}>
                            {factInfo?.arrival?.type === FactTypes.CANCELLED ||
                            allStationDirectionsCanceled ? (
                                DEFAULT_VALUE
                            ) : (
                                <>
                                    {!isNoStop &&
                                        formatTime({
                                            time: arrival,
                                            isFuzzy,
                                        })}
                                    {arrivalAndDepartureIsDifferentDay && (
                                        <div
                                            className={b('arrivalDate')}
                                            data-nosnippet
                                        >
                                            {arrivalDate &&
                                                arrivalDate.format(
                                                    formatArrDepDate,
                                                )}
                                        </div>
                                    )}
                                </>
                            )}
                        </div>
                    </div>
                </td>
            )}
            {needDisplayColumn({
                columnName: ThreadTableColumnName.stay,
                isIntervalThread,
                isSuburbanBus,
            }) && (
                <td className={b('stay')}>
                    <div className={b('wrapper')}>
                        <div className={b('wrapperInner')}>
                            {arrivalDate &&
                            departureDate &&
                            !isNoStop &&
                            !anyStationDirectionsCanceled ? (
                                <Duration
                                    duration={
                                        (departureDate as any) -
                                        (arrivalDate as any)
                                    }
                                    options={{
                                        short: EDurationShortening.ALWAYS,
                                    }}
                                />
                            ) : (
                                DEFAULT_VALUE
                            )}
                        </div>
                    </div>
                </td>
            )}
            {needDisplayColumn({
                columnName: ThreadTableColumnName.departure,
                isSuburbanBus,
                isIntervalThread,
            }) && (
                <td className={b('departure')}>
                    <div className={b('wrapper')}>
                        <div className={b('wrapperInner')}>
                            {factInfo?.departure?.type ===
                                FactTypes.CANCELLED ||
                            allStationDirectionsCanceled ? (
                                DEFAULT_VALUE
                            ) : (
                                <>
                                    {!isNoStop &&
                                        formatTime({
                                            time: isSuburbanBus
                                                ? departure || arrival
                                                : departure,
                                            isFuzzy,
                                        })}
                                    {arrivalAndDepartureIsDifferentDay && (
                                        <div
                                            className={b('departureDate')}
                                            data-nosnippet
                                        >
                                            {departureDate &&
                                                departureDate.format(
                                                    formatArrDepDate,
                                                )}
                                        </div>
                                    )}
                                </>
                            )}
                        </div>
                    </div>
                </td>
            )}
            {needDisplayColumn({
                columnName: ThreadTableColumnName.intervalInfo,
                isIntervalThread,
                isSuburbanBus,
            }) && (
                <td className={b('intervalInfo')}>
                    <div className={b('wrapper')}>
                        <div className={b('wrapperInner')}>
                            {displayIntervalThreadInfo &&
                                Boolean(beginTime) &&
                                Boolean(endTime) &&
                                !isNoStop &&
                                !allStationDirectionsCanceled && (
                                    <div className={b('intervalTime')}>
                                        {beginTime
                                            .split(':')
                                            .slice(0, 2)
                                            .join(':')}
                                        ...{' '}
                                        {endTime
                                            .split(':')
                                            .slice(0, 2)
                                            .join(':')}
                                    </div>
                                )}
                            {displayIntervalThreadInfo &&
                                !allStationDirectionsCanceled && (
                                    <div className={b('intervalDensity')}>
                                        {density}
                                    </div>
                                )}
                            {displayIntervalThreadInfo &&
                                !allStationDirectionsCanceled && (
                                    <div className={b('intervalComment')}>
                                        {comment}
                                    </div>
                                )}
                        </div>
                    </div>
                </td>
            )}
            {needDisplayColumn({
                columnName: ThreadTableColumnName.timeInTheWay,
                isIntervalThread,
                isSuburbanBus,
            }) && (
                <td className={b('timeInTheWay')}>
                    <div className={b('wrapper')}>
                        <div className={b('wrapperInner')}>
                            {!(
                                isNoStop ||
                                allStationDirectionsCanceled ||
                                (anyStationDirectionsCanceled &&
                                    !isIntermediateStation)
                            ) && (
                                <Duration
                                    duration={
                                        (arrivalDate as any) -
                                        (startDateOfThread as any)
                                    }
                                    defaultValue={DEFAULT_VALUE}
                                    options={{
                                        short: EDurationShortening.ALWAYS,
                                    }}
                                />
                            )}
                        </div>
                    </div>
                </td>
            )}
        </tr>
    );
}

function getRow(params: TGetRowParams): React.ReactElement | null {
    if (isSubtitleTableRowParams(params)) {
        return getSubtitle(params);
    }

    if (isTransitionTableRowParams(params)) {
        return getTransition(params);
    }

    if (isStationTableRowParams(params)) {
        return getStation(params);
    }

    return null;
}

interface IGButtonToggleStationsParams {
    colspan: number;
    prevStationsIsOpened: boolean;
    nextStationsIsOpened: boolean;
    transportType: TransportType;
    onClickTogglePrevStations: () => void;
    onClickToggleNextStations: () => void;

    type?: ThreadType;
}

function getButtonToggleStations({
    colspan,
    prevStationsIsOpened,
    nextStationsIsOpened,
    transportType,
    onClickTogglePrevStations,
    onClickToggleNextStations,

    type = ThreadType.prev,
}: IGButtonToggleStationsParams): React.ReactElement {
    const isPrev = type === ThreadType.prev;
    const className = isPrev
        ? b('buttonPrevStations', {isOpened: prevStationsIsOpened})
        : b('buttonNextStations', {isOpened: nextStationsIsOpened});
    const onClick = isPrev
        ? onClickTogglePrevStations
        : onClickToggleNextStations;
    const textButton = isPrev
        ? prevStationsIsOpened
            ? threadKeyset('hide-prev-stations')
            : threadKeyset('display-prev-stations')
        : nextStationsIsOpened
        ? threadKeyset('hide-next-stations')
        : threadKeyset('display-next-stations');
    const arrowDirection = isPrev
        ? prevStationsIsOpened
            ? 'up'
            : 'down'
        : nextStationsIsOpened
        ? 'up'
        : 'down';

    reachGoalOnce(
        `thread_${transportType}_station_${isPrev ? 'before' : 'after'}_show`,
    );

    return (
        <tr className={b('rowButton')}>
            <td colSpan={colspan}>
                <span className={className} onClick={onClick}>
                    <span className={b('buttonText')}>{textButton}</span>
                    <Arrow
                        className={b('buttonIcon')}
                        direction={arrowDirection}
                    />
                </span>
            </td>
        </tr>
    );
}

interface IThreadTableParams {
    stationFromData: IStation;
    stationToData: IStation;
    shouldShowTimezoneSwitch: boolean;
    stations: IStation[];
    prevStationsIsOpened: boolean;
    nextStationsIsOpened: boolean;
    isIntervalThread: boolean;
    isSuburbanBus: boolean;
    capitals: ICapital[];
    capitalSlug: string;
    fromStationDepartureLocalDt: string;
    id: string;
    canonicalUid: string;
    beginTime: string;
    endTime: string;
    density: string;
    comment: string;
    transportType: TransportType;

    departure?: string;
    departureFrom?: string;
    currentTimezone?: string;
    stationFrom?: string;
    stationTo?: string;
    isFullCanceling?: boolean;
    className?: string;
}

export default memo(ThreadTable);

function ThreadTable({
    stationFromData,
    stationToData,
    shouldShowTimezoneSwitch = false,
    stations,
    prevStationsIsOpened,
    nextStationsIsOpened,
    isIntervalThread,
    isSuburbanBus,
    capitals,
    capitalSlug,
    fromStationDepartureLocalDt,
    id,
    canonicalUid,
    beginTime,
    endTime,
    density,
    comment,
    transportType,

    stationFrom,
    stationTo,
    departure,
    departureFrom,
    currentTimezone,
    isFullCanceling,
    className,
}: IThreadTableParams): React.ReactElement {
    const tld = useSelector(state => state.tld);
    const language = useSelector(state => state.language);
    const dispatch = useDispatch();

    const shouldDisplaySwitchTimezones = Boolean(
        isMobile && shouldShowTimezoneSwitch,
    );
    const tableRows: TTableRowContent[] = [];

    const colspan = useMemo(() => {
        return Object.values(ThreadTableColumnName)
            .map(name =>
                needDisplayColumn({
                    columnName: name,
                    isIntervalThread,
                    isSuburbanBus,
                }),
            )
            .filter(Boolean).length;
    }, [isIntervalThread, isSuburbanBus]);

    const startDateOfThread = moment.parseZone(
        stationFromData.departureLocalDt,
    );

    let startDate: Moment | '' = moment
        .parseZone(stations[0].departureLocalDt)
        .add(-1, 'days');

    let thread: ThreadType = ThreadType.prev;
    let {country: prevCountry, capitalTimeOffset: prevCapitalTimeOffset} =
        stations[0];

    const arrivalKeyName = isMobile ? 'arrival-time-short' : 'arrival-time';
    const departureKeyName = isSuburbanBus
        ? 'departure-time-long'
        : isMobile
        ? 'departure-time-short'
        : 'departure-time';

    stations.forEach(station => {
        const isStationFrom = station === stationFromData;
        const isStationTo = station === stationToData;
        const firstShownStation = prevStationsIsOpened
            ? station === stations[0]
            : isStationFrom;

        if (isStationFrom) {
            thread = ThreadType.current;
        }

        const {
            arrivalLocalDt,
            departureLocalDt,
            country,
            transitionTime,
            capitalTimeOffset,
        } = station;

        const departureDate =
            departureLocalDt && moment.parseZone(departureLocalDt);
        const arrivalDate = arrivalLocalDt && moment.parseZone(arrivalLocalDt);
        const stationDate = arrivalDate || departureDate;

        // Для Украины не показываем пересечение гос. границы
        const isDifferentCountry =
            tld !== TLD_UA && country.id !== prevCountry.id;
        const timezoneDifference = capitalTimeOffset !== prevCapitalTimeOffset;
        const shouldShowTimeOffset = Boolean(
            !currentTimezone &&
                (firstShownStation ||
                    timezoneDifference ||
                    station === stations[0]),
        );
        const shouldShowDate = Boolean(
            !isIntervalThread &&
                (firstShownStation ||
                    (stationDate &&
                        startDate &&
                        startDate.format(ROBOT) !== stationDate.format(ROBOT))),
        );

        if (shouldShowDate || shouldShowTimeOffset || isDifferentCountry) {
            tableRows.push({
                type: ThreadTableRowType.subtitle,
                stationDate:
                    shouldShowDate && stationDate ? stationDate : undefined,
                country: isDifferentCountry ? country : undefined,
                capitalTimeOffset: shouldShowTimeOffset
                    ? capitalTimeOffset
                    : undefined,
                colspan,
                thread,
            });

            if (shouldShowDate) {
                startDate = stationDate && stationDate.clone();
            }
        }

        if (station.isCombined) {
            tableRows.push(
                {
                    type: ThreadTableRowType.station,
                    thread,
                    station,
                    arrivalDate: arrivalDate || undefined,
                },
                {
                    type: ThreadTableRowType.transition,
                    transitionTime,
                    colspan,
                    thread: isStationTo ? ThreadType.next : thread,
                },
                {
                    type: ThreadTableRowType.station,
                    thread: isStationTo ? ThreadType.next : thread,
                    station,
                    departureDate: departureDate || undefined,
                },
            );
        } else {
            tableRows.push({
                type: ThreadTableRowType.station,
                thread,
                station,
                departureDate: departureDate || undefined,
                arrivalDate: arrivalDate || undefined,
            });
        }

        if (station === stationToData) {
            thread = ThreadType.next;
        }

        prevCountry = country;
        prevCapitalTimeOffset = capitalTimeOffset;
    });

    const prevStations = tableRows.filter(row => row.thread === 'prev');
    const segmentStations = tableRows.filter(row => row.thread === 'current');
    const nextStations = tableRows.filter(row => row.thread === 'next');
    const firstStationIndex = segmentStations.findIndex(
        row => row.type === 'station',
    );

    const onClickTogglePrevStations = useCallback(() => {
        if (!prevStationsIsOpened) {
            reachGoal(`thread_${transportType}_station_before_click`);
        }

        return dispatch(setPrevStationsIsOpened(!prevStationsIsOpened));
    }, [prevStationsIsOpened, transportType, dispatch]);

    const onClickToggleNextStations = useCallback(() => {
        if (!nextStationsIsOpened) {
            reachGoal(`thread_${transportType}_station_after_click`);
        }

        return dispatch(setNextStationsIsOpened(!nextStationsIsOpened));
    }, [transportType, nextStationsIsOpened, dispatch]);

    const onClickRowStation = useCallback(
        (e: React.MouseEvent<HTMLElement>) => {
            if (!isMobile) {
                return;
            }

            reachGoal(`thread_${transportType}_station_click`);

            e.preventDefault();
            window.location.href = e.currentTarget.dataset.href || '';
        },
        [transportType],
    );

    const onClickStationName = useCallback(() => {
        reachGoal(`thread_${transportType}_station_click`);
    }, [transportType]);

    const getRowParams = (
        row: TTableRowContent,
        index: number,
    ): TGetRowParams => {
        if (isStationTableRow(row)) {
            return {
                row,
                index,
                startDateOfThread,
                stationFromData,
                stationToData,
                beginTime,
                endTime,
                density,
                comment,
                transportType,
                isSuburbanBus,
                isIntervalThread,
                tld,
                language,
                onClickRowStation,
                onClickStationName,
                isFullCanceling,
                stations,
            };
        }

        if (isSubtitleTableRow(row)) {
            return {
                row,
                index,
                capitals,
                capitalSlug,
            };
        }

        return {
            row,
            index,
        };
    };

    return (
        <div className={b({isMobile, isSuburbanBus}, className)}>
            <table className={b('table')}>
                <thead className={b('thead')}>
                    <tr>
                        {needDisplayColumn({
                            columnName: ThreadTableColumnName.station,
                            isIntervalThread,
                            isSuburbanBus,
                        }) && (
                            <th className={b('th', {thStation: true})}>
                                {threadKeyset('station')}
                            </th>
                        )}
                        {needDisplayColumn({
                            columnName: ThreadTableColumnName.arrival,
                            isIntervalThread,
                            isSuburbanBus,
                        }) && (
                            <th className={b('th')}>
                                {threadKeyset(arrivalKeyName)}
                            </th>
                        )}
                        {needDisplayColumn({
                            columnName: ThreadTableColumnName.stay,
                            isIntervalThread,
                            isSuburbanBus,
                        }) && (
                            <th className={b('th')}>
                                {threadKeyset('stay-time')}
                            </th>
                        )}
                        {needDisplayColumn({
                            columnName: ThreadTableColumnName.departure,
                            isIntervalThread,
                            isSuburbanBus,
                        }) && (
                            <th className={b('th')}>
                                {threadKeyset(departureKeyName)}
                            </th>
                        )}
                        {needDisplayColumn({
                            columnName: ThreadTableColumnName.intervalInfo,
                            isIntervalThread,
                            isSuburbanBus,
                        }) && (
                            <th className={b('th')}>
                                {threadKeyset('departure-time-long')}
                            </th>
                        )}
                        {needDisplayColumn({
                            columnName: ThreadTableColumnName.timeInTheWay,
                            isIntervalThread,
                            isSuburbanBus,
                        }) && (
                            <th className={b('th')}>
                                {threadKeyset('time-in-the-way')}
                            </th>
                        )}
                    </tr>
                    <tr>
                        <th
                            colSpan={colspan}
                            className={b('headerLineContainer')}
                        >
                            <div className={b('headerLine')} />
                        </th>
                    </tr>
                    {shouldDisplaySwitchTimezones && (
                        <tr>
                            <th
                                colSpan={colspan}
                                className={b('th', {timezoneSwitch: true})}
                            >
                                <ThreadTimezonesSwitch
                                    className={b('timezoneSwitchContainer')}
                                    mods={{selectWidth: 'full'}}
                                    id={id}
                                    canonicalUid={canonicalUid}
                                    capitals={capitals}
                                    stationFrom={stationFrom}
                                    stationTo={stationTo}
                                    departure={departure}
                                    departureFrom={departureFrom}
                                    currentTimezone={currentTimezone}
                                    fromStationDepartureLocalDt={
                                        fromStationDepartureLocalDt
                                    }
                                />
                            </th>
                        </tr>
                    )}
                    <tr>
                        <th colSpan={colspan} className={b('headerSpace')} />
                    </tr>
                </thead>
                <tbody
                    className={b('prevStations', {
                        isOpened: prevStationsIsOpened,
                    })}
                >
                    {prevStations.map((row, index) =>
                        getRow(getRowParams(row, index)),
                    )}
                </tbody>
                <tbody
                    className={b('segmentStations', {
                        prevStationsIsNone: prevStations.length === 0,
                    })}
                >
                    {Boolean(prevStations.length) &&
                        getButtonToggleStations({
                            colspan,
                            type: ThreadType.prev,
                            prevStationsIsOpened,
                            transportType,
                            onClickToggleNextStations,
                            onClickTogglePrevStations,
                            nextStationsIsOpened,
                        })}
                    {segmentStations.map((row, index) =>
                        getRow({
                            ...getRowParams(row, index),

                            displayIntervalThreadInfo:
                                index === firstStationIndex,
                        }),
                    )}
                    {Boolean(nextStations.length) &&
                        getButtonToggleStations({
                            colspan,
                            type: ThreadType.next,
                            prevStationsIsOpened,
                            transportType,
                            onClickToggleNextStations,
                            onClickTogglePrevStations,
                            nextStationsIsOpened,
                        })}
                </tbody>
                <tbody
                    className={b('nextStations', {
                        isOpened: nextStationsIsOpened,
                    })}
                >
                    {nextStations.map((row, index) =>
                        getRow(getRowParams(row, index)),
                    )}
                </tbody>
            </table>
            <div className={b('lastLine')} />
        </div>
    );
}
