import {React, PureComponent, B, PropTypes} from '../../base';

import moment from 'moment';
import upperFirst from 'lodash/upperFirst';
import page from 'page';
import {connect} from 'react-redux';

import {PROP_TYPES, DEFAULT_PROPS} from '../propTypes';

import Platform from '../../../interfaces/Platform';

import getThreadUrl from '../../../lib/url/getThreadUrl';

import Select from '../../basic/Select/Select';

import timezonesKeyset from '../../../i18n/timezones';

const b = B('ThreadTimezonesSwitch');

const mapStateToProps = ({tld, language}) => ({
    tld,
    language,
});

/*
mods:
selectWidth_full - селект будет занимать всю ширину блока
 */
class ThreadTimezonesSwitch extends PureComponent {
    static propTypes = {
        id: PROP_TYPES.id,
        canonicalUid: PROP_TYPES.canonicalUid,
        capitals: PROP_TYPES.capitals,
        stationFrom: PROP_TYPES.stationFrom,
        stationTo: PROP_TYPES.stationTo,
        departure: PROP_TYPES.departure,
        departureFrom: PROP_TYPES.departureFrom,
        fromStationDepartureLocalDt: PROP_TYPES.fromStationDepartureLocalDt,

        currentTimezone: PROP_TYPES.currentTimezone,
        className: PropTypes.string,
        mods: PropTypes.object,
    };

    static defaultProps = {
        currentTimezone: DEFAULT_PROPS.currentTimezone,
        className: '',
        mods: {},
    };

    onChange = option => {
        const url = option.value;
        const options = this.getOptions();
        const currentOption = this.getCurrentOption(options);

        if (currentOption.value === url) {
            return false;
        }

        page.show(url);
    };

    getOptions() {
        const {
            id,
            canonicalUid,
            capitals,
            stationFrom,
            stationTo,
            departure,
            departureFrom,
            fromStationDepartureLocalDt,
            tld,
            language,
        } = this.props;

        const threadUrlParams = {
            tld,
            language,
            departure,
            departureFromMoment:
                departureFrom && moment.parseZone(fromStationDepartureLocalDt),
            threadUid: id,
            canonicalUid,
            stationFromId: stationFrom,
            stationToId: stationTo,
        };

        const options = capitals.map(capital => {
            const {slug, title, titleGenitive, timezone} = capital;

            return {
                value: getThreadUrl({
                    ...threadUrlParams,
                    timezone,
                }),
                title:
                    timezonesKeyset.get(slug) ||
                    (titleGenitive &&
                        timezonesKeyset('time-by-city', {
                            cityNameGenitive: titleGenitive,
                        })) ||
                    timezonesKeyset('time-in-city', {cityName: title}),
                timezone,
            };
        });

        // Добавляем местное время в опции
        options.push({
            value: getThreadUrl(threadUrlParams),
            title: upperFirst(timezonesKeyset('local-time')),
            timezone: '',
        });

        return options;
    }

    getCurrentOption(options) {
        const {currentTimezone} = this.props;

        return options.find(({timezone}) => timezone === currentTimezone);
    }

    render() {
        const {className, mods} = this.props;

        const options = this.getOptions();
        const currentOption = this.getCurrentOption(options);

        return (
            <div className={b(mods, className)}>
                <span className={b('printText')}>
                    {timezonesKeyset('specified-time', {
                        title: currentOption.title,
                    })}
                </span>

                <Select
                    className={b('select')}
                    onChange={this.onChange}
                    value={currentOption}
                    options={options}
                    type="encircled"
                    listItemSize={
                        process.env.PLATFORM === Platform.mobile
                            ? 'big'
                            : undefined
                    }
                    titleTextAlign="center"
                />
            </div>
        );
    }
}

export default connect(mapStateToProps)(ThreadTimezonesSwitch);
