import React from 'react';
import B from 'bem-cn-lite';
import {connect} from 'react-redux';
import {momentTimezone as moment} from '../../../reexports';

import {HUMAN_MONTH_WITH_YEAR} from '../../lib/date/formats';
import {CHAR_EM_DASH} from '../../lib/stringUtils';
import {
    CHANGE,
    CURRENT,
    OTHER,
    NON_CONTEXT,
    ORDER_BASIC_CURRENT,
    ORDER_CHANGE_CURRENT,
    ORDER,
} from '../../lib/constants/threadScheduleTypes';
import {
    THREAD_SCHEDULE_TYPES,
    THREAD_SCHEDULE_TYPES_SIMPLIFIED,
} from '../../lib/transportType';

// eslint-disable-next-line no-duplicate-imports
import {TransportType} from '../../lib/transportType';
import FactTypes from '../../interfaces/state/stateThread/factInfo/FactTypes';

import {
    getStationFrom,
    getStationTo,
    getFuncDataForDayComponent,
} from '../../lib/thread/thread';
import {getActivePlainMask, getMonthsByMask} from '../../lib/date/mask';
import {reachGoal} from '../../lib/yaMetrika';
import isRidesharingEnabled from '../../lib/segments/ridesharing/isRidesharingEnabled';

import {
    setMapIsOpened,
    setMonthIndex,
    setScheduleBlockIsOpened,
} from '../../actions/thread';

import {SegmentBuyButtonUtmContextProvider} from '../../contexts/SegmentBuyButtonUtmContext';
import ThreadPageBase from './ThreadPageBase';
import TransportIcon from '../TransportIcon/TransportIcon';
import ThreadDates from './ThreadDates';
import ThreadInfo from './ThreadInfo';
import ThreadTable from './ThreadTable/ThreadTable';
import Month from '../Month/Month';
import ThreadPageCalendarDayLink from '../ThreadPageCalendarDayLink';
import Arrow from '../Arrow/Arrow';
import Link from '../Link';
import Button from '../Button/Button';
import ModalThreadMap from './ModalThreadMap';
import NoticeStar from '../NoticeStar';
import ThreadSchedule from './ThreadSchedule';
import OtherTodayThreads from './OtherTodayThreads';
import LoadingChunk from '../basic/LoadingChunk';
import RelatedThreads from './RelatedThreads';
import ThreadBottomDescription from './ThreadBottomDescription';
import ThreadBlaBlaCar from './ThreadBlaBlaCar';
import ThreadPrices from './ThreadPrices/ThreadPrices.desktop';

import threadKeyset from '../../i18n/thread';

const b = B('ThreadPage');

const buyButtonUtmMediumContext = {
    getUtmMedium: () => 'desktop_thread_button',
};

const mapStateToProps = ({tld, language, currencies, thread, page, flags}) => ({
    tld,
    language,
    currencies,
    thread,
    page,
    flags,
});

class ThreadPage extends ThreadPageBase {
    onClickChangeMonth = e => {
        const {dispatch} = this.props;
        const {monthIndex} = e.currentTarget.dataset;

        return dispatch(setMonthIndex(monthIndex));
    };

    onClickCalendarDate = () => {
        const {transportType} = this.props.thread;

        reachGoal(`thread_${transportType}_calendar_click`);
    };

    onClickShowScheduleBlock = () => {
        const {dispatch, thread} = this.props;
        const {scheduleBlockIsOpened} = thread;

        return dispatch(setScheduleBlockIsOpened(!scheduleBlockIsOpened));
    };

    onCloseMap = () => {
        const {dispatch} = this.props;

        return dispatch(setMapIsOpened(false));
    };

    onCompanyMailClick = () => {
        const {transportType} = this.props.thread;

        reachGoal(`thread_${transportType}_carrier_email_click`);
    };

    onCompanyWebsiteClick = () => {
        const {transportType} = this.props.thread;

        reachGoal(`thread_${transportType}_carrier_website_click`);
    };

    getThreadsByType({currentScheduleType, currentInContext}) {
        const {threads, transportType} = this.props.thread;
        const threadsByType = {};

        Object.entries(threads).forEach(([threadKey, threadInfo]) => {
            const {type, inContext, current} = threadInfo;

            if (!type) {
                return null;
            }

            const nonContextThread = currentInContext && !inContext;
            const scheduleBaseType = THREAD_SCHEDULE_TYPES_SIMPLIFIED.includes(
                transportType,
            )
                ? nonContextThread
                    ? NON_CONTEXT
                    : ''
                : type;
            const shouldAddSubtype =
                transportType === TransportType.suburban || !nonContextThread;
            const scheduleSubtype = shouldAddSubtype
                ? current
                    ? CURRENT
                    : currentScheduleType === type
                    ? OTHER
                    : ''
                : '';
            const scheduleType = `${scheduleBaseType}${
                scheduleBaseType && scheduleSubtype && '-'
            }${scheduleSubtype}`;

            if (threadsByType[scheduleType]) {
                threadsByType[scheduleType].push({
                    ...threadInfo,
                    key: threadKey,
                });
            } else {
                threadsByType[scheduleType] = [
                    {
                        ...threadInfo,
                        key: threadKey,
                    },
                ];
            }
        });

        return threadsByType;
    }

    getBlaBlaCar() {
        const {thread, currencies, language, flags} = this.props;
        const {blablacar} = thread;
        const {
            banned,
            querying,
            tariff,
            allDaysCheckComplete,
            allDaysCheckResult,
        } = blablacar || {};

        if (!isRidesharingEnabled(flags)) {
            return null;
        }

        // Это условие появилось в рамках задачи https://st.yandex-team.ru/RASPFRONT-7838
        // При возобновлении нормальной работы ББК нужно оторвать
        const noResultsOnDate =
            !tariff?.offersCount && Boolean(allDaysCheckResult);

        if (noResultsOnDate) {
            return null;
        }

        const isAvailable =
            !banned &&
            (querying ||
                (tariff && tariff.offersCount) ||
                !allDaysCheckComplete ||
                allDaysCheckResult !== null);

        return (
            <div className={b('ridesharingContainer', {hide: !isAvailable})}>
                <ThreadBlaBlaCar
                    className={b('ridesharing')}
                    querying={querying}
                    tariff={tariff || {}}
                    allDaysCheckResult={allDaysCheckResult}
                    allDaysCheckComplete={allDaysCheckComplete}
                    currencies={currencies}
                    language={language}
                />
            </div>
        );
    }

    getCompanyInfo() {
        const {thread} = this.props;
        const {title, strange, description, url, address, email} =
            thread.company;

        if (!this.shouldShowCompanyInfo()) {
            return null;
        }

        return (
            <div className={b('company')}>
                <div className={b('companyElement', {bold: true})}>
                    {threadKeyset('carrier')} {title}
                </div>

                {Boolean(url) && !strange && (
                    <div className={b('companyElement')}>
                        <Link href={url} onClick={this.onCompanyWebsiteClick}>
                            {url}
                        </Link>
                    </div>
                )}

                {Boolean(address) && !strange && (
                    <div className={b('companyElement')}>{address}</div>
                )}

                {Boolean(this.phones.length) && !strange && (
                    <div className={b('companyElement')}>
                        {this.phones.map((phone, index) => (
                            <span key={index}>
                                {phone}
                                <br />
                            </span>
                        ))}
                    </div>
                )}

                {Boolean(email) && !strange && (
                    <div className={b('companyElement')}>
                        <Link
                            href={`mailto:${email}`}
                            onClick={this.onCompanyMailClick}
                        >
                            {email}
                        </Link>
                    </div>
                )}

                {Boolean(description) && !strange && (
                    <div className={b('companyElement')}>{description}</div>
                )}
            </div>
        );
    }

    // https://wiki.yandex-team.ru/Raspisanija/selector/ - описание логики построения селектора
    getThreadsSchedule({
        currentScheduleType,
        threadsByType,
        stationFromData: {title: stationFromTitle, id: stationFromId},
        stationToData: {title: stationToTitle, id: stationToId},
    }) {
        const {transportType, canonicalUid, currentTimezone} =
            this.props.thread;

        if (!THREAD_SCHEDULE_TYPES.includes(transportType)) {
            return null;
        }

        const threadsSchedule = [];
        let scheduleOrder;

        if (THREAD_SCHEDULE_TYPES_SIMPLIFIED.includes(transportType)) {
            scheduleOrder = ORDER;
        } else {
            scheduleOrder =
                currentScheduleType === CHANGE
                    ? ORDER_CHANGE_CURRENT
                    : ORDER_BASIC_CURRENT;
        }

        scheduleOrder.forEach(threadType => {
            if (threadsByType[threadType]) {
                threadsSchedule.push(
                    <ThreadSchedule
                        key={threadType}
                        type={threadType}
                        canonicalUid={canonicalUid}
                        threads={threadsByType[threadType]}
                        transportType={transportType}
                        stationFromId={stationFromId}
                        stationFromTitle={stationFromTitle}
                        stationToId={stationToId}
                        stationToTitle={stationToTitle}
                        currentTimezone={currentTimezone}
                    />,
                );
            }
        });

        return threadsSchedule;
    }

    renderLoading() {
        return <LoadingChunk />;
    }

    render() {
        const {tld, language, page, thread} = this.props;
        const {
            fetching,
            location: {query},
        } = page;

        const {
            id,
            shortTitle,
            number,
            canonicalUid,
            stationFrom,
            stationTo,
            departure,
            departureFrom,
            stations,
            transportType,
            runDays,
            runDaysText,
            threads,
            otherTodayThreads,
            relatedThreads,
            calendarIsOpened,
            prevStationsIsOpened,
            nextStationsIsOpened,
            monthIndex,
            isIntervalThread,
            isSuburbanBus,
            isNoChangeWagon,
            beginTime,
            endTime,
            density,
            comment,
            mapData,
            mapIsOpened,
            capitalSlug,
            capitals,
            fromStationDepartureLocalDt,
            currentTimezone,
            scheduleBlockIsOpened,
            isAeroExpress,
            isExpress,
            isToCitySearchContext,
        } = thread;

        if (fetching || !stations.length) {
            return this.renderLoading();
        }

        const shouldShowTimezoneSwitch = this.shouldShowTimezoneSwitch();
        const stationFromData = getStationFrom({stations});
        const stationToData = getStationTo({
            stations,
            stationFrom: stationFromData,
        });

        if (!stationFromData || !stationToData) {
            throw new Error(
                `Не удалось определить станцию ${
                    !stationFromData ? 'отправления' : 'прибытия'
                }.
                Data: ${JSON.stringify(this.props)}`,
            );
        }

        const dateStart = moment.parseZone(stationFromData.departureLocalDt);
        const dateStop = moment.parseZone(stationToData.arrivalLocalDt);
        const utcOffset = dateStart.utcOffset();
        const now = moment().utcOffset(utcOffset);
        const isSetStationTo = Boolean(query.station_to);
        const currentInContext = Boolean(query.station_from);

        const currentScheduleKey = Object.keys(threads).find(
            threadKey => threads[threadKey].current,
        );
        const {type: currentScheduleType} = threads[currentScheduleKey] || {};
        const threadsByType = this.getThreadsByType({
            currentScheduleType,
            currentInContext,
        });
        const skipKeys = threadsByType.cancel
            ? threadsByType.cancel.map(threadInfo => Number(threadInfo.key))
            : [];
        const datesInCalendar = getActivePlainMask({
            mask: runDays,
            skipKeys,
            utcOffset,
        });
        const shouldColorCalendar =
            THREAD_SCHEDULE_TYPES.includes(transportType);

        const months = getMonthsByMask({mask: runDays, utcOffset});
        const displayMonthIndex = !months[monthIndex]
            ? months.findIndex(month => month.isSame(dateStart, 'month'))
            : monthIndex;
        const disabledPrevMonth = displayMonthIndex <= 0;
        const disabledNextMonth = displayMonthIndex === months.length - 1;
        const needShowScheduleBlock =
            (transportType === TransportType.train && scheduleBlockIsOpened) ||
            (transportType !== TransportType.bus &&
                transportType !== TransportType.train);

        const isFullCanceling =
            stationFromData?.factInfo?.departure?.type ===
                FactTypes.CANCELLED &&
            stationToData?.factInfo?.arrival?.type === FactTypes.CANCELLED;

        return (
            <SegmentBuyButtonUtmContextProvider
                value={buyButtonUtmMediumContext}
            >
                <div
                    className={b({
                        showTimezoneSwitch: shouldShowTimezoneSwitch,
                    })}
                >
                    <div className={b('header')}>
                        <TransportIcon
                            className={b('transportIcon')}
                            transportType={transportType}
                            isExpress={isExpress}
                            isAeroExpress={isAeroExpress}
                            size="giant"
                        />
                        <h1 className={b('title')}>{this.title}</h1>
                    </div>
                    {transportType === TransportType.train && (
                        <ThreadBottomDescription
                            number={number}
                            shortTitle={shortTitle}
                            stations={stations}
                        />
                    )}

                    <div className={b('scheduleText')}>
                        {threadKeyset('current-schedule')}:{' '}
                        <span data-nosnippet>{runDaysText}</span>
                    </div>

                    <div className={b('daysContainer')}>
                        <ThreadDates
                            id={id}
                            canonicalUid={canonicalUid}
                            runDays={runDays}
                            skipKeys={skipKeys}
                            stations={stations}
                            isSetStationTo={isSetStationTo}
                            calendarIsOpened={calendarIsOpened}
                            countDatesInline={7}
                            transportType={transportType}
                            currentTimezone={currentTimezone}
                            isToCitySearchContext={isToCitySearchContext}
                        />
                    </div>

                    <div className={b('content')}>
                        <div className={b('leftColumn')} id="threadPrintArea">
                            <ThreadInfo
                                className={b('threadInfo')}
                                id={id}
                                canonicalUid={canonicalUid}
                                capitals={capitals}
                                stationFrom={stationFrom}
                                stationTo={stationTo}
                                departure={departure}
                                departureFrom={departureFrom}
                                dateStart={dateStart}
                                dateStop={dateStop}
                                mapData={mapData}
                                capitalSlug={capitalSlug}
                                currentTimezone={currentTimezone}
                                transportType={transportType}
                                shouldShowTimezoneSwitch={
                                    shouldShowTimezoneSwitch
                                }
                                idPrintBlock="threadPrintArea"
                                fromStationDepartureLocalDt={
                                    fromStationDepartureLocalDt
                                }
                            />

                            <ThreadTable
                                className={b('table')}
                                id={id}
                                canonicalUid={canonicalUid}
                                stations={stations}
                                stationFrom={stationFrom}
                                stationTo={stationTo}
                                stationFromData={stationFromData}
                                stationToData={stationToData}
                                prevStationsIsOpened={prevStationsIsOpened}
                                nextStationsIsOpened={nextStationsIsOpened}
                                isIntervalThread={isIntervalThread}
                                isSuburbanBus={isSuburbanBus}
                                beginTime={beginTime}
                                endTime={endTime}
                                density={density}
                                comment={comment}
                                capitalSlug={capitalSlug}
                                transportType={transportType}
                                capitals={capitals}
                                currentTimezone={currentTimezone}
                                shouldShowTimezoneSwitch={
                                    shouldShowTimezoneSwitch
                                }
                                fromStationDepartureLocalDt={
                                    fromStationDepartureLocalDt
                                }
                                isFullCanceling={isFullCanceling}
                            />
                            {stations.find(
                                station => station.isTechnicalStop,
                            ) && (
                                <div className={b('notice')}>
                                    <NoticeStar />
                                    <span className={b('noticeMessage')}>
                                        {` ${CHAR_EM_DASH} ${threadKeyset(
                                            'technical-stop',
                                        )}`}
                                    </span>
                                </div>
                            )}
                        </div>

                        <div className={b('rightColumn')}>
                            <div className={b('monthContainer')}>
                                <Button
                                    className={b('prevMonth')}
                                    disabled={disabledPrevMonth}
                                    onClick={this.onClickChangeMonth}
                                    data-month-index={displayMonthIndex - 1}
                                >
                                    <Arrow
                                        className={b('arrow')}
                                        direction="left"
                                    />
                                </Button>

                                <span className={b('monthName')}>
                                    {months[displayMonthIndex]
                                        .format(HUMAN_MONTH_WITH_YEAR)
                                        .toUpperCase()}
                                </span>

                                <Button
                                    className={b('nextMonth')}
                                    disabled={disabledNextMonth}
                                    onClick={this.onClickChangeMonth}
                                    data-month-index={displayMonthIndex + 1}
                                >
                                    <Arrow
                                        className={b('arrow')}
                                        direction="right"
                                    />
                                </Button>
                            </div>

                            <Month
                                className={b('month')}
                                today={now}
                                dateStart={dateStart}
                                month={months[displayMonthIndex]}
                                showWeekdays
                                showMonthName={false}
                                language={language}
                                onClick={this.onClickCalendarDate}
                                DayComponent={ThreadPageCalendarDayLink}
                                getDataForDayComponent={getFuncDataForDayComponent(
                                    {
                                        tld,
                                        language,
                                        stationFromId: stationFromData.id,
                                        stationToId: isSetStationTo
                                            ? stationToData.id
                                            : null,
                                        datesInCalendar,
                                        runDays,
                                        threads,
                                        threadId: id,
                                        canonicalUid,
                                        stations,
                                        shouldColorCalendar,
                                        timezone: currentTimezone,
                                        isToCitySearchContext,
                                    },
                                )}
                            />

                            <ThreadPrices className={b('price')} />

                            {transportType === TransportType.train && (
                                <div className={b('showScheduleBlock')}>
                                    <span
                                        onClick={this.onClickShowScheduleBlock}
                                    >
                                        {scheduleBlockIsOpened
                                            ? threadKeyset.get(
                                                  'hide-detailed-schedule',
                                              )
                                            : threadKeyset.get(
                                                  'show-detailed-schedule',
                                              )}
                                    </span>
                                </div>
                            )}

                            {needShowScheduleBlock && (
                                <div className={b('scheduleBlock')}>
                                    {Boolean(otherTodayThreads.length) && (
                                        <OtherTodayThreads
                                            otherTodayThreads={
                                                otherTodayThreads
                                            }
                                            canonicalUid={canonicalUid}
                                            transportType={transportType}
                                            stationFromId={stationFromData.id}
                                            stationToId={stationToData.id}
                                            currentTimezone={currentTimezone}
                                        />
                                    )}
                                    {this.getThreadsSchedule({
                                        currentScheduleType,
                                        threadsByType,
                                        stationToData,
                                        stationFromData,
                                    })}
                                    {Boolean(
                                        Object.keys(relatedThreads).length,
                                    ) && (
                                        <RelatedThreads
                                            relatedThreads={relatedThreads}
                                            transportType={transportType}
                                            isNoChangeWagon={isNoChangeWagon}
                                            currentTimezone={currentTimezone}
                                        />
                                    )}
                                </div>
                            )}
                            {this.getCompanyInfo()}
                            <div>{this.getBlaBlaCar()}</div>
                        </div>
                    </div>

                    {mapData && (
                        <ModalThreadMap
                            mapData={mapData}
                            transportType={transportType}
                            onClose={this.onCloseMap}
                            visible={mapIsOpened}
                        />
                    )}
                </div>
            </SegmentBuyButtonUtmContextProvider>
        );
    }
}

export default connect(mapStateToProps)(ThreadPage);
