import React from 'react';
import {connect} from 'react-redux';
import B from 'bem-cn-lite';

import {momentTimezone as moment} from '../../../reexports';

import {CHAR_EM_DASH} from '../../lib/stringUtils';

import {TransportType} from '../../lib/transportType';
import FactTypes from '../../interfaces/state/stateThread/factInfo/FactTypes';

import {getStationFrom, getStationTo} from '../../lib/thread/thread';
import {reachGoal} from '../../lib/yaMetrika';
import isRidesharingEnabled from '../../lib/segments/ridesharing/isRidesharingEnabled';

import {setCompanyInfoIsOpened} from '../../actions/thread';

import {SegmentBuyButtonUtmContextProvider} from '../../contexts/SegmentBuyButtonUtmContext';
import ThreadPageBase from './ThreadPageBase';
import TransportIcon from '../TransportIcon/TransportIcon';
import SheetCompany from './SheetCompany';
import ThreadDates from './ThreadDates';
import LoadingChunk from '../basic/LoadingChunk';
import ThreadInfo from './ThreadInfo';
import ThreadTable from './ThreadTable/ThreadTable';
import NoticeStar from '../NoticeStar';
import ToTheTop from '../ToTheTop';
import RidesharingSegment from '../RidesharingSegment';
import ThreadBottomDescription from './ThreadBottomDescription';
import ThreadPriceConainer from './ThreadPriceConainer/ThreadPriceConainer.mobile';

import commonKeyset from '../../i18n/common';
import threadKeyset from '../../i18n/thread';
import timezonesKeyset from '../../i18n/timezones';

const b = B('ThreadPage');

const buyButtonUtmMediumContext = {
    getUtmMedium: () => 'thread_button',
};

const mapStateToProps = ({
    language,
    page,
    thread,
    currencies,
    flags,
    user,
}) => ({
    language,
    page,
    thread,
    currencies,
    flags,
    user,
});

class ThreadPage extends ThreadPageBase {
    onClickContacts = () => {
        const {dispatch, thread} = this.props;
        const {companyInfoIsOpened, transportType, company} = thread;

        reachGoal(`thread_${transportType}_carrier_contact_click`);
        this.reachGoalCompanyInfo({
            company,
            phones: this.phones,
            transportType,
        });

        dispatch(setCompanyInfoIsOpened(!companyInfoIsOpened));
    };

    onCloseContacts = () => {
        const {dispatch} = this.props;

        dispatch(setCompanyInfoIsOpened(false));
    };

    getRidesharingSegment() {
        const {thread, language, currencies, flags} = this.props;
        const {blablacar} = thread;
        const {
            banned,
            querying,
            tariff,
            allDaysCheckComplete,
            allDaysCheckResult,
        } = blablacar || {};

        if (!isRidesharingEnabled(flags)) {
            return null;
        }

        // Это условие появилось в рамках задачи https://st.yandex-team.ru/RASPFRONT-7838
        // При возобновлении нормальной работы ББК нужно оторвать
        const noResultsOnDate =
            !tariff?.offersCount && Boolean(allDaysCheckResult);

        if (noResultsOnDate) {
            return null;
        }

        const isAvailable =
            !banned &&
            (querying ||
                (tariff && tariff.offersCount) ||
                !allDaysCheckComplete ||
                allDaysCheckResult !== null);

        return (
            <div className={b('ridesharingContainer', {hide: !isAvailable})}>
                <RidesharingSegment
                    className={b('ridesharing')}
                    querying={querying}
                    tariff={tariff}
                    allDaysCheckResult={allDaysCheckResult}
                    allDaysCheckComplete={allDaysCheckComplete}
                    currencies={currencies}
                    language={language}
                />
            </div>
        );
    }

    renderLoading() {
        return <LoadingChunk />;
    }

    render() {
        const {page, thread} = this.props;

        const {
            fetching,
            location: {query},
        } = page;

        const {
            id,
            shortTitle,
            number,
            canonicalUid,
            stations,
            stationFrom,
            stationTo,
            departure,
            departureFrom,
            transportType,
            company,
            runDays,
            runDaysText,
            companyInfoIsOpened,
            calendarIsOpened,
            mapData,
            prevStationsIsOpened,
            nextStationsIsOpened,
            isIntervalThread,
            isSuburbanBus,
            beginTime,
            endTime,
            density,
            comment,
            segment,
            capitalSlug,
            capitals,
            fromStationDepartureLocalDt,
            currentTimezone,
            isExpress,
            isAeroExpress,
            isToCitySearchContext,
        } = thread;

        const shouldShowTimezoneSwitch = this.shouldShowTimezoneSwitch();

        if (fetching || !stations.length) {
            return this.renderLoading();
        }

        const stationFromData = getStationFrom({stations});
        const stationToData = getStationTo({
            stations,
            stationFrom: stationFromData,
        });

        if (!stationFromData || !stationToData) {
            throw new Error(
                `Не удалось определить станцию ${
                    !stationFromData ? 'отправления' : 'прибытия'
                }.
                Data: ${JSON.stringify(this.props)}`,
            );
        }

        const dateStart = moment.parseZone(stationFromData.departureLocalDt);
        const dateStop = moment.parseZone(stationToData.arrivalLocalDt);
        const shouldShowCompanyInfo = this.shouldShowCompanyInfo();
        const isFullCanceling =
            stationFromData?.factInfo?.departure?.type ===
                FactTypes.CANCELLED ||
            stationToData?.factInfo?.arrival?.type === FactTypes.CANCELLED;

        return (
            <SegmentBuyButtonUtmContextProvider
                value={buyButtonUtmMediumContext}
            >
                <div
                    className={b({
                        showTimezoneSwitch: shouldShowTimezoneSwitch,
                    })}
                >
                    <div className={b('head')}>
                        <div className={b('transport')}>
                            <TransportIcon
                                className={b('transportIcon')}
                                transportType={transportType}
                                isExpress={isExpress}
                                isAeroExpress={isAeroExpress}
                                size="huge"
                            />
                        </div>

                        <h1 className={b('title')}>{this.title}</h1>
                        {transportType === TransportType.train && (
                            <ThreadBottomDescription
                                number={number}
                                segment={segment}
                                shortTitle={shortTitle}
                                stations={stations}
                            />
                        )}

                        <div className={b('scheduleText')}>
                            {threadKeyset('current-schedule')}:{' '}
                            <span data-nosnippet>{runDaysText}</span>
                        </div>

                        {company.title && !company.hidden && (
                            <div className={b('company')}>
                                <span className={b('companyName')}>
                                    {company.title}
                                </span>
                                {shouldShowCompanyInfo && (
                                    <span
                                        className={b('companyContactsButton')}
                                        onClick={this.onClickContacts}
                                    >
                                        {commonKeyset('contacts')}
                                    </span>
                                )}
                            </div>
                        )}

                        <div className={b('daysContainer')}>
                            <ThreadDates
                                id={id}
                                canonicalUid={canonicalUid}
                                runDays={runDays}
                                stations={stations}
                                calendarIsOpened={calendarIsOpened}
                                transportType={transportType}
                                isSetStationTo={Boolean(query.station_to)}
                                isToCitySearchContext={isToCitySearchContext}
                                currentTimezone={currentTimezone}
                            />
                        </div>
                    </div>

                    <div className={b('priceContainer')}>
                        <ThreadPriceConainer priceClassName={b('price')} />
                    </div>

                    <ThreadInfo
                        className={b('threadInfo')}
                        dateStart={dateStart}
                        dateStop={dateStop}
                        mapData={mapData}
                        transportType={transportType}
                    />

                    <div className={b('tableContainer')}>
                        {!shouldShowTimezoneSwitch && (
                            <div className={b('localTime')}>
                                {timezonesKeyset('local-time-specified')}
                            </div>
                        )}
                        <ThreadTable
                            className={b('table')}
                            id={id}
                            canonicalUid={canonicalUid}
                            stations={stations}
                            stationFrom={stationFrom}
                            stationTo={stationTo}
                            stationFromData={stationFromData}
                            stationToData={stationToData}
                            departure={departure}
                            departureFrom={departureFrom}
                            prevStationsIsOpened={prevStationsIsOpened}
                            nextStationsIsOpened={nextStationsIsOpened}
                            isIntervalThread={isIntervalThread}
                            isSuburbanBus={isSuburbanBus}
                            beginTime={beginTime}
                            endTime={endTime}
                            density={density}
                            comment={comment}
                            capitalSlug={capitalSlug}
                            transportType={transportType}
                            capitals={capitals}
                            currentTimezone={currentTimezone}
                            shouldShowTimezoneSwitch={shouldShowTimezoneSwitch}
                            fromStationDepartureLocalDt={
                                fromStationDepartureLocalDt
                            }
                            isFullCanceling={isFullCanceling}
                        />
                        {stations.find(station => station.isTechnicalStop) && (
                            <div className={b('notice')}>
                                <NoticeStar />
                                <span className={b('noticeMessage')}>
                                    {` ${CHAR_EM_DASH} ${threadKeyset(
                                        'technical-stop',
                                    )}`}
                                </span>
                            </div>
                        )}
                    </div>

                    <div className={b('bottomSpace')} />

                    <ToTheTop className={b('toTheTop')} />

                    {shouldShowCompanyInfo && (
                        <SheetCompany
                            className={b('sheetCompany')}
                            transportType={transportType}
                            isOpened={companyInfoIsOpened}
                            onClose={this.onCloseContacts}
                            onClickOutside={this.onCloseContacts}
                            phones={this.phones}
                            {...company}
                        />
                    )}
                </div>
            </SegmentBuyButtonUtmContextProvider>
        );
    }
}

export default connect(mapStateToProps)(ThreadPage);
