import {React, PureComponent, PropTypes, mix, B} from '../../base';

import {connect} from 'react-redux';

import {PAGE} from '../propTypes';

import IconGlyph from '../../../interfaces/components/IconGlyph';

import {stationUrl} from '../../../lib/url/stationUrl';
import buildTransportCityUrl from '../../../lib/url/transportCityUrl';
import getSubtypeByTransportType from '../../../lib/station/getSubtypeByTransportType';

import Link from '../../Link';
import Icon from '../../Icon/Icon';

import keyset from '../../../i18n/transport';

const b = B('TransportCities');

const isMobile = process.env.PLATFORM === 'mobile';

const mapStateToProps = ({tld, language}) => ({
    tld,
    language,
});

class TransportCities extends PureComponent {
    static propTypes = {
        className: PropTypes.string,
        transportType: PAGE.transportType,
        cities: PAGE.cities,
        mods: PropTypes.object,
    };

    static defaultProps = {
        className: '',
        mods: {},
    };

    getCityHref(city, transportType) {
        const {slug, stationId, pageType, mainSubtype} = city;
        const {tld, language} = this.props;

        if (slug) {
            return buildTransportCityUrl({
                slug,
                transportType,
            });
        }

        return stationUrl({
            id: stationId,
            pageType,
            mainSubtype,
            subtype: getSubtypeByTransportType(transportType),
            isMobile,
            tld,
            language,
        });
    }

    render() {
        const {mods, cities, className, transportType} = this.props;

        return (
            <div className={mix(b({isMobile, ...mods}), className)}>
                <div className={b('title')}>
                    <Icon
                        className={b('icon', {transportType})}
                        glyph={IconGlyph.pointIcon}
                    />
                    <div className={b('titleText')}>
                        {keyset(`cities-${transportType}`)}
                    </div>
                </div>
                <div>
                    {cities.map(city => (
                        <div key={city.slug || city.stationId}>
                            <Link
                                key={city.slug}
                                className={b('city')}
                                href={this.getCityHref(city, transportType)}
                            >
                                <Icon
                                    className={b('pinIcon')}
                                    glyph={IconGlyph.pinLite}
                                />
                                {city.title}
                            </Link>
                        </div>
                    ))}
                </div>
            </div>
        );
    }
}

export default connect(mapStateToProps)(TransportCities);
