import {React, PureComponent, PropTypes, mix, B} from '../../base';

import {connect} from 'react-redux';

import {PAGE} from '../propTypes';

import {stationUrl} from '../../../lib/url/stationUrl';
import buildTransportCityUrl from '../../../lib/url/transportCityUrl';
import getSubtypeByTransportType from '../../../lib/station/getSubtypeByTransportType';

import Link from '../../Link';
import TransportIconInCircle from '../../TransportIconInCircle/TransportIconInCircle';

import keyset from '../../../i18n/transport';

const b = B('TransportCity');

const isMobile = process.env.PLATFORM === 'mobile';

const mapStateToProps = ({tld, language}) => ({
    tld,
    language,
});

class TransportCity extends PureComponent {
    static propTypes = {
        className: PropTypes.string,
        mods: PropTypes.object,
        transportType: PAGE.transportType,
        city: PropTypes.oneOfType([PAGE.mainCity, PAGE.secondaryCity]),
    };

    static defaultProps = {
        className: '',
        mods: {},
    };

    render() {
        const {className, mods, transportType, tld, language} = this.props;

        const {slug, title, stations, titleGenitive} = this.props.city;

        return (
            <div className={mix(b({isMobile, ...mods}), className)}>
                <div className={b('title')}>
                    <TransportIconInCircle
                        className={b('icon', {transportType})}
                        transportType={transportType}
                    />
                    <div className={b('titleText')}>
                        <Link
                            href={buildTransportCityUrl({slug, transportType})}
                        >
                            {keyset(`stations-${transportType}`, {
                                titleGenitive: titleGenitive || title,
                            })}
                        </Link>
                    </div>
                </div>
                <div className={b('list')}>
                    {stations.map(station => (
                        <div key={station.id}>
                            <Link
                                className={b('link')}
                                href={stationUrl({
                                    id: station.id,
                                    pageType: station.pageType,
                                    subtype:
                                        getSubtypeByTransportType(
                                            transportType,
                                        ),
                                    mainSubtype: station.mainSubtype,
                                    isMobile,
                                    tld,
                                    language,
                                })}
                            >
                                {station.title}
                            </Link>
                        </div>
                    ))}
                </div>
            </div>
        );
    }
}

export default connect(mapStateToProps)(TransportCity);
