import {React, PureComponent, B} from '../base';

import {connect} from 'react-redux';

import {PAGE} from './propTypes';

import LoadingChunk from '../basic/LoadingChunk';
import TransportCity from './TransportCity';
import TransportCities from './TransportCities';
import TransportArticles from './TransportArticles';
import TransportCountries from './TransportCountries';

import keyset from '../../i18n/transport';

const b = B('TransportPage');
const SECONDARY_CITY_TITLE_OFFSET = 2;
const COUNTRY_TITLE_OFFSET = 1;

// пока не показываем страны
const showCountries = false;

const mapsStateToProps = ({tld}) => ({tld});

class TransportPage extends PureComponent {
    static propTypes = PAGE;

    renderLoading() {
        return <LoadingChunk />;
    }

    separateCities() {
        const {cities, mainCity, countries, secondaryCity} = this.props;

        const secondaryStationsLength = secondaryCity.stations.length;
        const mainStationsLength = mainCity.stations.length;
        const citiesLength = cities.length;
        const countriesLength = showCountries ? countries.length - 1 : 0;

        let separatorIndex;
        let offset = 0;

        const allItemsSeparatedByHalf = Math.floor(
            (secondaryStationsLength + citiesLength + countriesLength) / 2,
        );

        if (secondaryStationsLength > 0) {
            offset = showCountries
                ? COUNTRY_TITLE_OFFSET
                : SECONDARY_CITY_TITLE_OFFSET;
        }

        if (allItemsSeparatedByHalf <= mainStationsLength) {
            separatorIndex = mainStationsLength - secondaryStationsLength;
        } else {
            separatorIndex = allItemsSeparatedByHalf - secondaryStationsLength;
        }

        return [
            cities.slice(0, separatorIndex - offset),
            cities.slice(separatorIndex - offset),
        ];
    }

    render() {
        const {
            tld,
            page,
            cities,
            articles,
            mainCity,
            countries,
            secondaryCity,
            transportType,
        } = this.props;

        if (page.fetching) {
            return this.renderLoading();
        }

        const hasMainCity = mainCity && mainCity.stations.length > 0;
        const hasSecondaryCity =
            secondaryCity && secondaryCity.stations.length > 0;
        const hasArticles = articles && articles.length > 0;
        const hasCountries = countries && countries.length > 0;
        const hasCities = cities && cities.length > 0;

        const bannerHref = `${process.env.IMAGES_PATH}/transportBanners/${transportType}.jpg`;
        const linearGradient = `
            linear-gradient(
                to bottom,
                rgba(0, 0, 0, 0) 0%,
                rgba(0, 0, 0, 0) 50%,
                rgba(0, 0, 0, 0.7) 100%
            )
        `;
        const bannerStyle = {
            background: `${linearGradient}, url(${bannerHref}), #777`,
            backgroundSize: 'cover',
            backgroundPositionY: 'center',
        };

        const filtredCountries = countries.filter(
            country => country.code.toLowerCase() !== tld,
        );

        const [firstColumn, secondColumn] = this.separateCities();

        return (
            <div className={b()}>
                <div className={b('content')}>
                    <div
                        style={bannerStyle}
                        className={b('banner', {transportType})}
                    >
                        <div className={b('bannerText')}>
                            {keyset(`banner-${transportType}`)}
                        </div>
                    </div>
                    <div className={b('main')}>
                        {hasMainCity && (
                            <TransportCity
                                className={b('city')}
                                city={mainCity}
                                transportType={transportType}
                            />
                        )}
                        <div className={b('firstCitiesColumn')}>
                            {hasSecondaryCity && (
                                <TransportCity
                                    className={b('secondaryCity')}
                                    city={secondaryCity}
                                    transportType={transportType}
                                />
                            )}
                            {hasCities && (
                                <TransportCities
                                    cities={firstColumn}
                                    transportType={transportType}
                                />
                            )}
                        </div>
                        <div className={b('secondCitiesColumn')}>
                            {hasCities && secondColumn.length > 0 && (
                                <TransportCities
                                    className={b('cities')}
                                    cities={secondColumn}
                                    mods={{withoutTitle: true}}
                                    transportType={transportType}
                                />
                            )}
                            {showCountries && hasCountries && (
                                <TransportCountries
                                    className={b('countries')}
                                    countries={filtredCountries}
                                    transportType={transportType}
                                />
                            )}
                        </div>
                    </div>
                </div>
                {hasArticles && (
                    <TransportArticles
                        className={b('articles')}
                        articles={articles}
                    />
                )}
            </div>
        );
    }
}

export default connect(mapsStateToProps)(TransportPage);
