import {PureComponent, PropTypes} from '../base';

import getElementRect from '../../lib/dimensions/getElementRect';

import {
    setTooltipContent,
    setTooltipPosition,
    setTooltipVisibility,
} from '../../actions/tooltip';

let tooltipTimeout = null;
const DELAY = 100;

const defaultProps = {
    offsetX: 0,
    offsetY: 4,
};

export default class TooltipComponent extends PureComponent {
    static propTypes = {
        hint: PropTypes.string.isRequired, // текст в попапе
        offsetX: PropTypes.number,
        offsetY: PropTypes.number,
    };

    static defaultProps = defaultProps;

    onMouseEnter = e => {
        if (tooltipTimeout) {
            clearTimeout(tooltipTimeout);
        }

        const {dispatch} = this.props;
        const {offsetX = defaultProps.offsetX, offsetY = defaultProps.offsetY} =
            this.props;
        const rect = getElementRect(e.currentTarget);

        tooltipTimeout = setTimeout(() => {
            dispatch(setTooltipContent(this.getHint()));
            dispatch(
                setTooltipPosition({
                    top: rect.bottom + offsetY,
                    left: rect.left + offsetX,
                }),
            );
            dispatch(setTooltipVisibility(true));
            tooltipTimeout = null;
        }, DELAY);
    };

    onMouseLeave = e => {
        if (e.relatedTarget === window) {
            /*
            Это багфикс события mouseleave в IE11: если событие навешено на элемент,
            содержащий в себе svg, использующий use, то событие будет генерится на каждый
            выход за пределы любого элемента внутри svg, как будто это событие mouseout.
            https://github.com/facebook/react/issues/13115
             */
            return;
        }

        const {dispatch} = this.props;

        if (tooltipTimeout) {
            clearTimeout(tooltipTimeout);
            dispatch(setTooltipVisibility(false));
        } else {
            tooltipTimeout = setTimeout(() => {
                dispatch(setTooltipVisibility(false));
                tooltipTimeout = null;
            }, DELAY);
        }
    };

    getHint() {
        return this.props.hint;
    }
}
