import {React, Component, B, refCallback, PropTypes} from '../../base';

import throttle from 'lodash/throttle';

import noop from '../../../lib/noop';

const b = B('BottomSheet');

export default class BottomSheet extends Component {
    static propTypes = {
        onClose: PropTypes.func.isRequired,
        opened: PropTypes.bool.isRequired,

        onClickOutside: PropTypes.func,
    };

    static defaultProps = {
        onClickOutside: noop,
    };

    state = {
        isMoving: false,
        position: 0,
        shift: 0,
        rendered: false,
    };

    componentDidMount() {
        import('hammerjs').then(() => this._initHammer());
    }

    componentWillReceiveProps(nextProps) {
        const {opened} = this.props;
        const {opened: nextOpened} = nextProps;

        if (opened !== nextOpened) {
            if (nextOpened === true) {
                document.addEventListener('click', this.onClickOutside);
            } else {
                document.removeEventListener('click', this.onClickOutside);
            }
        }
    }

    componentWillUnmount() {
        document.removeEventListener('click', this.onClickOutside);
    }

    onPanEnd = () => {
        const {onClose} = this.props;
        const position = Math.max(this.state.shift, 0);
        const closed = position > this._content.offsetHeight / 4;

        this.setState({
            shift: 0,
            position: 0,
            isMoving: false,
        });

        if (closed) {
            onClose();
        }
    };

    onPanVertical = e => {
        this.setState({
            shift: e.deltaY,
            isMoving: true,
        });
    };

    onClickOutside = e => {
        const {opened, onClickOutside} = this.props;

        if (opened && !this._content.contains(e.target)) {
            onClickOutside();
        }
    };

    _initHammer() {
        this.hammer = new Hammer(this._self);
        this.hammer.get('pan').set({direction: Hammer.DIRECTION_VERTICAL});
        this.hammer.on(
            'pandown panup',
            throttle(this.onPanVertical, 16, {
                leading: false,
                trailing: false,
            }),
        );
        this.hammer.on('panend', this.onPanEnd);
        setTimeout(() => this.setState({rendered: true}), 50);
    }

    getContentStyle() {
        const {shift, position} = this.state;
        const transform = `translateY(${Math.max(position + shift, 0)}px)`;

        return {
            transform,
            // Фикс для особо древних
            WebkitTransform: transform,
        };
    }

    render() {
        const {rendered, isMoving} = this.state;
        const {opened, children} = this.props;

        if (!children) {
            return null;
        }

        return (
            <div className={b()} ref={refCallback(this, '_self')}>
                <div
                    className={b('content', {
                        isMoving,
                        closed: !opened || !rendered,
                    })}
                    ref={refCallback(this, '_content')}
                    style={this.getContentStyle()}
                >
                    {children}
                </div>
            </div>
        );
    }
}
