import {React, Component, mix, PropTypes, B} from '../../base';

import IconGlyph from '../../../interfaces/components/IconGlyph';

import noop from '../../../lib/noop';

import RootContext from '../../Root/RootContext';
import Icon from '../../Icon/Icon';
import Popup from '../../Popup/Popup';
import Button from '../../Button/Button';

export const b = B('Modal');

export default class Modal extends Component {
    static propTypes = {
        visible: PropTypes.bool,
        title: PropTypes.string,
        text: PropTypes.string,
        fixed: PropTypes.bool,
        className: PropTypes.string,
        hasCloseButton: PropTypes.bool,
        onClose: PropTypes.func,
        onShow: PropTypes.func,
        mods: PropTypes.object,
    };

    static defaultProps = {
        visible: false,
        title: '',
        text: '',
        fixed: true,
        className: '',
        hasCloseButton: true,
        onClose: noop,
        onShow: noop,
        mods: {},
    };

    componentWillReceiveProps(nextProps) {
        if (nextProps.visible !== this.props.visible) {
            this.setBackgroundIsFixed(nextProps.visible);
        }
    }

    componentWillUnmount() {
        this.setBackgroundIsFixed(false);
    }

    onClose = () => {
        this.props.onClose();
        this.setBackgroundIsFixed(false);
    };

    onOpen = () => {
        this.props.onShow();
        this.setBackgroundIsFixed(true);
    };

    static contextType = RootContext;

    setBackgroundIsFixed(value) {
        if (this.props.fixed) {
            if (value) {
                this.context.fixBody();
            } else {
                this.context.releaseBody();
            }
        }
    }

    render() {
        const {
            visible,
            hasCloseButton,
            title,
            text,
            children,
            className,
            mods,
        } = this.props;

        return (
            <Popup
                onClickOutside={this.onClose}
                visible={visible}
                onShow={this.onOpen}
                className={mix(b(mods), className)}
            >
                <div className={b('content')}>
                    {title && <h3 className={b('title')}>{title}</h3>}

                    {text && <p className={b('text')}>{text}</p>}

                    {children}
                </div>

                {hasCloseButton && (
                    <Button className={b('closeButton')} onClick={this.onClose}>
                        <Icon
                            className={b('closeIcon')}
                            glyph={IconGlyph.cross}
                        />
                    </Button>
                )}
            </Popup>
        );
    }
}
