import React, {
    ReactElement,
    useState,
    useRef,
    useCallback,
    SyntheticEvent,
    ChangeEvent,
    useEffect,
    useMemo,
} from 'react';
import B from 'bem-cn-lite';

import KeyCode from '../../../interfaces/KeyCode';
import SecondaryPosition from '../../../interfaces/lib/dimensions/SecondaryPosition';
import PrimaryPosition from '../../../interfaces/lib/dimensions/PrimaryPosition';
import Platform from '../../../interfaces/Platform';
import IconGlyph from '../../../interfaces/components/IconGlyph';

import {makeCacheable} from '../../../lib/cache';
import typedMemo from '../../typedMemo';

import Icon from '../../Icon/Icon';
import Arrow from '../../Arrow/Arrow';
import Popup from '../../Popup/Popup';
import Button from '../../Button/Button';
import Button2 from '../../Button2/Button2';

const b = B('Select');

const isMobile = process.env.PLATFORM === Platform.mobile;

const POPUP_POSITIONS = [
    [PrimaryPosition.bottom, SecondaryPosition.left],
    [PrimaryPosition.bottom, SecondaryPosition.right],
    [PrimaryPosition.top, SecondaryPosition.left],
    [PrimaryPosition.top, SecondaryPosition.right],
];

type Value = string | number | boolean;

type Option = {
    value: Value;
    title: string;

    disabled?: boolean;
};

interface ISelect<T extends Option> {
    options: T[];
    onChange: (option: T) => void;

    className?: string;
    value?: T;
    defaultValue?: T;
    // encircled - селект с обводкой по контуру, как выбор таймзоны на странице нитки
    // grayButton - серый селект
    type?: 'encircled' | 'grayButton';
    popupTheme?: 'travel'; // Оформление Popup
    listItemSize?: 'big'; // big - большой размер пунктов селекта (используется на таче)
    ellipsis?: boolean; // Текст внутри может обрезаться с "..." на конце
    minWidth?: number; // Минимальная ширина кнопки. Имеет смысл использовать совместно с "ellipsis: true"
    name?: string;
    formatTitle?: (option?: T) => string; // Позволяет форматировать отображаемый на кнопке текст выбранного значения
    titleTextAlign?: 'left' | 'center' | 'right';
    popupPositions?: [PrimaryPosition, SecondaryPosition][];
    onFocus?: () => void;
    onBlur?: () => void;
    onOpen?: () => void;
    onClose?: () => void;
    onClick?: () => void;
    native?: boolean; // Если true, то будет использован нативный селект (барабан на мобильных устройствах). Для тачей по умолчанию true, для десктопа - false
}

const getOnItemClick = makeCacheable(
    <T extends Option>(option: T, setSelectedOption: ISelect<T>['onChange']) =>
        () => {
            setSelectedOption(option);
        },
);

const getOnItemMouseOver = makeCacheable(
    <T extends Option>(option: T, setHoverOption: (option: T) => void) =>
        () => {
            setHoverOption(option);
        },
);

export default typedMemo(Select);

function Select<T extends Option>({
    options,
    onChange,

    className,
    value,
    defaultValue,
    type,
    popupTheme = 'travel',
    listItemSize,
    ellipsis,
    minWidth,
    formatTitle,
    titleTextAlign,
    popupPositions,
    onFocus: onFocusFromProps,
    onBlur: onBlurFromProps,
    onOpen,
    onClose,
    onClick,
    native = isMobile,
}: ISelect<T>): ReactElement {
    const selectedOption = value || defaultValue;

    const [opened, setOpened] = useState(false);
    const [focused, setFocused] = useState(false);
    const [hoverOption, setHoverOption] = useState(selectedOption);

    const domElement = useRef<HTMLDivElement | null>(null);

    const open = useCallback(() => {
        setOpened(true);

        if (onOpen) {
            onOpen();
        }
    }, [onOpen]);

    const close = useCallback(() => {
        setOpened(false);

        if (onClose) {
            onClose();
        }
    }, [onClose]);

    const onClickOutside = useCallback(
        (e: SyntheticEvent) => {
            const {target} = e;
            const {current: selectDomElement} = domElement;

            if (!selectDomElement) {
                return;
            }

            if (
                selectDomElement !== target &&
                target instanceof Node &&
                !selectDomElement.contains(target)
            ) {
                close();
            }
        },
        [close],
    );

    const onClickTitle = useCallback(
        (e: SyntheticEvent) => {
            e.preventDefault();

            if (onClick) {
                onClick();
            }

            if (opened) {
                close();
            } else {
                open();
            }
        },
        [close, open, opened, onClick],
    );

    const setSelectedOption = useCallback(
        (option: T) => {
            if (option.disabled) {
                return;
            }

            setHoverOption(option);
            close();
            onChange(option);
        },
        [onChange, close],
    );

    const onChangeNative = useCallback(
        (event: ChangeEvent<HTMLSelectElement>) => {
            const index = parseInt(event.target.value, 10);

            setSelectedOption(options[index]);
        },
        [options, setSelectedOption],
    );

    const onFocus = useCallback(() => {
        setFocused(true);

        if (onFocusFromProps) {
            onFocusFromProps();
        }
    }, [onFocusFromProps]);

    const onBlur = useCallback(() => {
        setFocused(false);

        if (onBlurFromProps) {
            onBlurFromProps();
        }
    }, [onBlurFromProps]);

    const onKeyPress = useCallback(
        (e: KeyboardEvent): void => {
            e.preventDefault();

            const pressedKey = e.which || e.keyCode;

            switch (pressedKey) {
                case KeyCode.enter:
                    if (
                        opened &&
                        hoverOption &&
                        hoverOption !== selectedOption
                    ) {
                        setSelectedOption(hoverOption);
                    }

                    break;
                case KeyCode.esc:
                    close();
                    break;
                case KeyCode.up: {
                    if (!opened) {
                        return open();
                    }

                    const hoverIndex = hoverOption
                        ? options.findIndex(option => option === hoverOption)
                        : 0;
                    const newHoverIndex =
                        hoverIndex > 0 ? hoverIndex - 1 : options.length - 1;

                    setHoverOption(options[newHoverIndex]);

                    break;
                }

                case KeyCode.down: {
                    if (!opened) {
                        return open();
                    }

                    const hoverIndex = hoverOption
                        ? options.findIndex(option => option === hoverOption)
                        : 0;
                    const newHoverIndex =
                        hoverIndex >= 0 && hoverIndex < options.length - 1
                            ? hoverIndex + 1
                            : 0;

                    setHoverOption(options[newHoverIndex]);

                    break;
                }

                default: {
                    return;
                }
            }
        },
        [
            close,
            hoverOption,
            open,
            opened,
            options,
            selectedOption,
            setSelectedOption,
        ],
    );

    useEffect(() => {
        if (focused) {
            document.addEventListener('keydown', onKeyPress);
        }

        return () => {
            document.removeEventListener('keydown', onKeyPress);
        };
    }, [focused, onKeyPress]);

    const selectedTitle = formatTitle
        ? formatTitle(selectedOption)
        : selectedOption
        ? selectedOption.title
        : '';

    const button = useMemo(() => {
        if (type === 'encircled' || type === 'grayButton') {
            return (
                <Button2
                    className={b('title')}
                    theme={type === 'encircled' ? 'select' : 'normal'}
                    iconRight={
                        <Icon
                            className={b('icon')}
                            glyph={IconGlyph.selectArrows}
                        />
                    }
                    onClick={onClickTitle}
                    ellipsis={ellipsis}
                    minWidth={minWidth}
                    textAlign={titleTextAlign}
                >
                    {selectedTitle}
                </Button2>
            );
        }

        return (
            <Button
                className={b('title')}
                type="button"
                rightIcon={
                    <Arrow
                        className={b('arrow')}
                        direction={opened ? 'up' : 'down'}
                    />
                }
                onClick={onClickTitle}
            >
                {selectedTitle}
            </Button>
        );
    }, [
        ellipsis,
        minWidth,
        onClickTitle,
        opened,
        selectedTitle,
        titleTextAlign,
        type,
    ]);

    const style = useMemo(() => ({minWidth}), [minWidth]);

    const optionIndex = options.findIndex(option => option === selectedOption);
    const selectedOptionIndex = optionIndex !== -1 ? optionIndex : undefined;

    return (
        <div
            className={b(
                {
                    type,
                    listItemSize,
                    opened,
                    focused,
                    ellipsis,
                    native,
                },
                className,
            )}
            onFocus={onFocus}
            onBlur={onBlur}
            ref={domElement}
            style={style}
        >
            {button}

            {!native && (
                <Popup
                    visible={opened}
                    onClickOutside={onClickOutside}
                    positions={popupPositions || POPUP_POSITIONS}
                    theme={popupTheme}
                    minContentWidth="full"
                    withoutArrow
                >
                    {options.map(option => {
                        const {value: optionValue, title, disabled} = option;
                        const itemMods = {
                            selected: optionValue === selectedOption?.value,
                            hovered: option === hoverOption,
                            disabled,
                        };
                        const onMouseOver =
                            process.env.PLATFORM === Platform.desktop
                                ? getOnItemMouseOver(option, setHoverOption)
                                : undefined;

                        return (
                            <div
                                key={`${optionValue}-${title}`}
                                className={b('item', itemMods)}
                                onClick={getOnItemClick(
                                    option,
                                    setSelectedOption,
                                )}
                                onMouseOver={onMouseOver}
                            >
                                <span className={b('checkContainer')}>
                                    {itemMods.selected ? (
                                        <Icon
                                            className={b('check')}
                                            glyph={IconGlyph.tick}
                                        />
                                    ) : null}
                                </span>

                                <span className={b('itemText')}>{title}</span>
                            </div>
                        );
                    })}
                </Popup>
            )}

            <select
                className={b('htmlSelect')}
                value={selectedOptionIndex}
                onChange={onChangeNative}
                disabled={!native}
            >
                {options.map(({title, disabled}, index) => (
                    <option
                        key={`${index}-${title}`}
                        value={index}
                        disabled={disabled}
                    >
                        {title}
                    </option>
                ))}
            </select>
        </div>
    );
}
