import {React, Component, refCallback, B} from '../../base';

import {subscribe, unsubscribe} from 'pubsub-js';
import {connect} from 'react-redux';

import {SEGMENTS} from '../../../lib/constants/pubSubEvents';
import {
    FULL_INFO,
    PARTNER_PICKER,
    AERO_INFO,
    RENDER_COMPONENT,
} from '../../../lib/constants/bottomSheetTypes';

import isAllDaysSearch from '../../../lib/search/isAllDaysSearch';
import {sortTariffClassKeys} from '../../../lib/segments/getBaseTariffClassKeys';

import SegmentFullInfo from '../SegmentFullInfo';
import SegmentCalendar from '../SegmentCalendar';
import BottomSheet from '../../basic/BottomSheet';
import ModalChoosePartner from '../../ModalChoosePartner';

import aeroexpressKeyset from '../../../i18n/bt-lib-aeroexpress';

const {
    SEGMENT_INFO_CLICK,
    SEGMENT_PARTNER_PICKER_CLICK,
    HOMEPAGE_AERO_INFO,
    UNIVERSAL,
} = SEGMENTS;

const b = B('SegmentBottomSheet');

const mapStateToProps = ({flags, language, tld, seoQueryParams}) => ({
    flags,
    language,
    tld,
    seoQueryParams,
});

class SegmentBottomSheet extends Component {
    state = {
        segment: null,
        klass: null,
        sheetIsOpened: false,
        calendarIsOpened: false,
        staticSheet: false,
        component: null,
    };

    componentDidMount() {
        this.pubSubEvents = {};
        this.pubSubEvents[SEGMENT_INFO_CLICK] = subscribe(
            SEGMENT_INFO_CLICK,
            this.onSegmentInfoClick,
        );
        this.pubSubEvents[SEGMENT_PARTNER_PICKER_CLICK] = subscribe(
            SEGMENT_PARTNER_PICKER_CLICK,
            this.onSegmentPartnerPickerClick,
        );
        this.pubSubEvents[HOMEPAGE_AERO_INFO] = subscribe(
            HOMEPAGE_AERO_INFO,
            this.onAeroInfoClick,
        );
        this.pubSubEvents[UNIVERSAL] = subscribe(
            UNIVERSAL,
            this.onUniversalClick,
        );

        document.addEventListener('click', this.onTouchOutside);
        document.addEventListener('scroll', this.onScroll);
    }

    componentWillUnmount() {
        document.removeEventListener('click', this.onTouchOutside);
        document.removeEventListener('scroll', this.onScroll);

        Object.values(this.pubSubEvents).forEach(unsubscribe);
        this.pubSubEvents = {};
    }

    onScroll = () => {
        this.closeSheet();
    };

    onTouchOutside = e => {
        const {sheetIsOpened} = this.state;

        if (sheetIsOpened) {
            if (!this._container.contains(e.target)) {
                this.onCloseTimeout = setTimeout(() => this.closeSheet(), 50);
            }
        }
    };

    onSegmentInfoClick = (message, {segment}) => {
        const {segments, isTransfer} = segment;
        const isTransferMainSegment = isTransfer && segments;

        if (isTransferMainSegment) {
            return;
        }

        clearInterval(this.onCloseTimeout);
        this.setState(state => ({
            segment,
            contentType: FULL_INFO,
            sheetIsOpened: state.sheetIsOpened
                ? state.segment !== segment
                : true,
        }));
    };

    onSegmentPartnerPickerClick = (message, {segment, klass}) => {
        clearInterval(this.onCloseTimeout);
        this.setState(state => ({
            segment,
            klass,
            contentType: PARTNER_PICKER,
            sheetIsOpened: state.sheetIsOpened
                ? state.segment !== segment ||
                  this.state.contentType !== PARTNER_PICKER
                : true,
        }));
    };

    onAeroInfoClick = () => {
        clearInterval(this.onCloseTimeout);
        this.setState(state => ({
            staticSheet: true,
            contentType: AERO_INFO,
            sheetIsOpened: !state.sheetIsOpened,
        }));
    };

    onCalendarOpen = () => {
        this.setState({
            calendarIsOpened: true,
        });
    };

    onCalendarClose = () => {
        this.setState({
            calendarIsOpened: false,
        });
    };

    onSheetClose = () => {
        this.closeSheet();
    };

    onUniversalClick = (message, {component}) => {
        clearInterval(this.onCloseTimeout);
        this.setState(state => ({
            staticSheet: true,
            contentType: RENDER_COMPONENT,
            sheetIsOpened: !state.sheetIsOpened,
            component,
        }));
    };

    closeSheet() {
        this.setState({
            sheetIsOpened: false,
        });
    }

    shouldRenderCalendar() {
        const {segment, contentType} = this.state;
        const {context} = this.props;

        return Boolean(
            segment && isAllDaysSearch(context) && contentType === FULL_INFO,
        );
    }

    getCalendar() {
        const {sort, context, flags, language, currencies} = this.props;
        const {segment, calendarIsOpened} = this.state;

        const tariffClassKeys = sortTariffClassKeys({
            tariffClassKeys: segment.tariffClassKeys || [],
            segment,
            sort,
        });

        return (
            <SegmentCalendar
                context={context}
                segment={segment}
                tariffClassKeys={tariffClassKeys}
                flags={flags}
                language={language}
                currencies={currencies}
                opened={calendarIsOpened}
                onClose={this.onCalendarClose}
            />
        );
    }

    getSheetContent() {
        const {segment, klass, contentType, component} = this.state;

        const {sort, context, currencies, isSuburbanSearchResult} = this.props;

        switch (contentType) {
            case FULL_INFO:
                return (
                    <SegmentFullInfo
                        sort={sort}
                        segment={segment}
                        context={context}
                        currencies={currencies}
                        onDatePickerOpen={this.onCalendarOpen}
                        isSuburbanSearchResult={isSuburbanSearchResult}
                    />
                );
            case PARTNER_PICKER:
                return (
                    <ModalChoosePartner tariffClass={klass} segment={segment} />
                );
            case AERO_INFO:
                return (
                    <span className={b('disclaimer-text')}>
                        {aeroexpressKeyset('disclaimer-description')}
                    </span>
                );
            case RENDER_COMPONENT:
                return component;

            default:
                return null;
        }
    }

    render() {
        const {segment, sheetIsOpened, staticSheet} = this.state;

        if (!segment && !staticSheet) {
            return null;
        }

        return (
            <div className={b()} ref={refCallback(this, '_container')}>
                <BottomSheet opened={sheetIsOpened} onClose={this.onSheetClose}>
                    {this.getSheetContent()}
                </BottomSheet>

                {this.shouldRenderCalendar() && this.getCalendar()}
            </div>
        );
    }
}

export default connect(mapStateToProps)(SegmentBottomSheet);
