import React, {PureComponent} from 'react';
import B from 'bem-cn-lite';
import {connect} from 'react-redux';

import {
    BUS_TYPE,
    PLANE_TYPE,
    TRAIN_TYPE,
    SUBURBAN_TYPE,
} from '../../../lib/transportType';
import {TRANSPORT_FEATURES} from '../../../lib/segments/transportFeatures/transportFeatures';

import IconGlyph from '../../../interfaces/components/IconGlyph';

import getSubtype from '../../../lib/segments/transportFeatures/getSubtype';
import getTransportFeatures from '../../../lib/segments/transportFeatures/getTransportFeatures';
import hasElectronicRegistration from '../../../lib/segments/transportFeatures/hasElectronicRegistration';

import Icon from '../../Icon/Icon';
import SegmentTransport from '../../SegmentTransport/SegmentTransport';
import SuburbanFacilities from '../../SuburbanFacilities/SuburbanFacilities';
import SegmentStops from '../../SegmentStops/SegmentStops';

import segmentInfoKeyset from '../../../i18n/segment-info';

const b = B('SegmentTypeInfo');
const TRAIN_FEATURES = TRANSPORT_FEATURES[TRAIN_TYPE];

function getModel(segment) {
    const {model} = segment.transport;

    if (model && model.title) {
        return <div className={b('model')}>{model.title}</div>;
    }

    return null;
}

const mapStateToProps = ({tld, language}) => ({tld, language});

class SegmentTypeInfo extends PureComponent {
    getBusInfo(segment) {
        const info = [];
        const model = getModel(segment);

        if (model) {
            info.push(model);
        }

        return info;
    }

    getPlaneInfo(segment) {
        const info = [];
        const model = getModel(segment);

        if (model) {
            info.push(model);
        }

        return info;
    }

    getTrainInfo(segment) {
        const {tld, language, isSuburbanSearchResult} = this.props;
        const {
            transport: {code: transportType},
            tariffs,
            hasDynamicPricing,
        } = segment;
        const info = [];
        const features = getTransportFeatures(
            segment,
            true,
            tld,
            language,
            isSuburbanSearchResult,
        ).filter(({type}) => TRAIN_FEATURES.includes(type));

        if (features.length) {
            info.push(
                <SegmentTransport
                    className={b('typeFeature')}
                    features={features}
                    transportType={transportType}
                    key="segment-transport"
                />,
            );
        }

        if (
            hasElectronicRegistration(
                transportType,
                tariffs && tariffs.electronicTicket,
            )
        ) {
            info.push(
                <span className={b('typeFeature')} key="e-ticket">
                    {segmentInfoKeyset('e-ticket')}
                </span>,
            );
        }

        if (hasDynamicPricing) {
            info.push(
                <span className={b('typeFeature')} key="dynamic-pricing">
                    {segmentInfoKeyset('dynamic-pricing')}
                </span>,
            );
        }

        return info;
    }

    getSuburbanInfo(segment) {
        const {isSuburbanSearchResult} = this.props;
        const info = [];
        const {stops, suburbanFacilities, transport, thread} = segment;
        const {code: transportType} = transport;
        const subtype = getSubtype(transport, thread, isSuburbanSearchResult);

        if (stops) {
            info.push(<SegmentStops className={b('stops')} stops={stops} />);
        }

        if (subtype) {
            info.push(
                <SegmentTransport
                    className={b('subtype')}
                    features={[subtype]}
                    transportType={transportType}
                />,
            );
        }

        if (suburbanFacilities && suburbanFacilities.length) {
            info.push(
                <SuburbanFacilities
                    expanded
                    suburbanFacilities={suburbanFacilities}
                />,
            );
        }

        return info;
    }

    getTypeInfo(transportType, segment) {
        switch (transportType) {
            case BUS_TYPE:
                return this.getBusInfo(segment);
            case PLANE_TYPE:
                return this.getPlaneInfo(segment);
            case TRAIN_TYPE:
                return this.getTrainInfo(segment);
            case SUBURBAN_TYPE:
                return this.getSuburbanInfo(segment);
        }

        return [];
    }

    getComment(segment) {
        const {comment} = segment.thread || {};

        if (!comment) {
            return null;
        }

        return (
            <div className={b('comment')} key="comment">
                <div className={b('commentTitle')}>
                    <Icon
                        className={b('commentIcon')}
                        glyph={IconGlyph.comment}
                    />

                    {segmentInfoKeyset('comment-title')}
                </div>

                <div className={b('commentText')}>{comment}</div>
            </div>
        );
    }

    render() {
        const {segment} = this.props;
        const {code: transportType} = segment.transport;

        const infoParts = this.getTypeInfo(transportType, segment);
        const comment = this.getComment(segment);

        if (comment) {
            infoParts.push(comment);
        }

        if (!infoParts.length) {
            return null;
        }

        return <div className={b()}>{infoParts}</div>;
    }
}

export default connect(mapStateToProps)(SegmentTypeInfo);
