import {useState, useEffect} from 'react';

import useForceUpdate from './useForceUpdate';

// Хук для получения данных.
// За данными умеет ходить функция getData, а задача этого хука дождаться получения этих данных,
// а когда дождется перерендерить компонент с учетом полученных данных
export default function useGetData<Input, Data>(
    input: Input,
    getData: (input: Input) => Data | Promise<Data>,
): Data | undefined {
    const forceUpdate = useForceUpdate();

    // Объект со всеми данными (по сути кеш без какаих-либо ограничений)
    const [allData] = useState<Map<Input, {result?: Data}>>(new Map());

    let data = allData.get(input);
    const callGetData = data ? undefined : getData(input);

    // Данных в кеше не было, но вызов функции для получения данных, сразу вернул результат, а не промис
    if (callGetData && !isPromise<Data>(callGetData)) {
        data = {result: callGetData};

        allData.set(input, data);
    }

    // Инициализируем поход за данными
    useEffect(() => {
        if (isPromise<Data>(callGetData)) {
            allData.set(input, {});

            callGetData
                .then(result => {
                    const dataObject = allData.get(input);

                    if (!dataObject) {
                        return;
                    }

                    dataObject.result = result;

                    // Производим ререндер
                    forceUpdate();
                })
                .catch(() => {
                    allData.delete(input);
                });
        }
    }, [allData, forceUpdate, callGetData, input]);

    return data?.result;
}

function isPromise<Data>(value: any): value is Promise<Data> {
    return Boolean(typeof value?.then === 'function');
}
