import EventEmitter from 'events';

import IQueueFactoryOfPromise from '../interfaces/lib/IQueueFactoryOfPromise';
import FactoryOfPromise from '../interfaces/FactoryOfPromise';

export interface IOptions {
    maxConcurrent?: number;
    maxLength?: number;
}

/*
Очередь, в которую можно класть фабрики промисов, и они будут обрабатываться параллельно,
с учетом опции maxConcurrent
*/
export default class QueueFactoryOfPromise<T = any>
    extends EventEmitter
    implements IQueueFactoryOfPromise<T>
{
    // Возвращает id для текущей задачи
    private static getIdForTask(): symbol {
        return Symbol('id');
    }

    protected options: Required<IOptions>;
    protected queue: FactoryOfPromise<T>[] = [];
    protected nowProcessing: symbol[] = [];
    protected countProcessed = 0;

    constructor(options: IOptions = {}) {
        super();
        this.options = {
            maxConcurrent: options.maxConcurrent ?? 1,
            maxLength: options.maxLength ?? 1000,
        };

        if (this.options.maxConcurrent < 1) {
            throw new Error(
                'Options "maxConcurrent" mast be greater then null',
            );
        }

        if (this.options.maxLength < 1) {
            throw new Error('Options "maxLength" mast be greater then null');
        }
    }

    // Возвращает длину очереди
    public getLength(): number {
        return this.queue.length + this.getProcessingLength();
    }

    // Возвращает количество задач, которые выполняются в данный момент
    public getProcessingLength(): number {
        return this.nowProcessing.length;
    }

    // Количество выполненных задач
    public getProcessedLength(): number {
        return this.countProcessed;
    }

    // Добавляем фабрику промиса в очередь
    public add(factoryOfPromise: FactoryOfPromise): number {
        if (this.getLength() > this.options.maxLength) {
            throw new Error(
                `Достигнут предел длины очереди (${this.options.maxLength})`,
            );
        }

        this.queue.push(factoryOfPromise);
        const length = this.getLength();

        this.emit('add', length);
        this.process();

        return length;
    }

    // Обрабатывает задачу из очереди
    public process(): boolean {
        if (this.getProcessingLength() >= this.options.maxConcurrent) {
            return false;
        }

        const factoryPromise = this.queue.shift();

        if (!factoryPromise) {
            return false;
        }

        const idTask = QueueFactoryOfPromise.getIdForTask();

        this.nowProcessing.push(idTask);
        factoryPromise()
            .then((result: any) => {
                this.emit('result', result);
            })
            .catch(error => {
                this.emit('error', error);
            })
            .finally(() => {
                this.countProcessed++;
                this.nowProcessing = this.nowProcessing.filter(
                    id => id !== idTask,
                );

                if (this.getLength() === 0) {
                    this.emit('end');
                } else {
                    this.process();
                }
            });

        return true;
    }
}
