import {TransportType} from './transportType';
import IStation from '../interfaces/state/stationsGroup/IStation';
import Tld from '../interfaces/Tld';
import Lang from '../interfaces/Lang';
import {
    hasStationAeroexpress,
    IStationWithAeroexpress,
} from '../interfaces/state/stationsGroup/IStationWithAeroexpress';

import makeUrl from './url/makeUrl';

const TRUE_AEROEXPRESS_AIRPORT_IDS = [
    9600215, // Внуково
    9600216, // Домодедово
    9600213, // Шереметьево
];

// Выбирает из списка аэропортов те, в которых есть аэроэкспресс
export function filterAirports(
    airports: IStation[],
): IStationWithAeroexpress[] {
    return airports.filter(hasStationAeroexpress);
}

// Решает, стоит ли показывать аэроэкспресс
export function shouldShowAeroexpress(
    stations: IStation[],
    type: TransportType,
): boolean {
    return type === TransportType.plane && filterAirports(stations).length > 0;
}

export function trueAeroexpress(airports: IStation[]): boolean {
    return airports.some(airport =>
        TRUE_AEROEXPRESS_AIRPORT_IDS.includes(airport.id),
    );
}

// Возвращает ссылку на поиск расписания аэроэкспрессов
export function getUrlForAirport(
    airport: IStationWithAeroexpress,
    isTrueAeroexpress: boolean,
    when: string,
    tld: Tld,
    language: Lang,
): string;
export function getUrlForAirport(
    airport: IStation,
    isTrueAeroexpress: boolean,
    when: string,
    tld: Tld,
    language: Lang,
): string | null;
export function getUrlForAirport(
    airport: IStation | IStationWithAeroexpress,
    isTrueAeroexpress: boolean,
    when: string,
    tld: Tld,
    language: Lang,
): string | null {
    if (!hasStationAeroexpress(airport)) {
        return null;
    }

    return makeUrl('/search/suburban/', tld, language, {
        fromId: `c${airport.aeroexpress.settlement_id}`,
        toId: `s${airport.id}`,
        aeroex: isTrueAeroexpress ? 'y' : '',
        when,
    });
}

// Возвращает ссылку на список аэроэкспрессов для заданного города
export function getAeroexpressListUrl(
    cityId: number,
    tld: Tld,
    language: Lang,
): string {
    return makeUrl('/stations/plane/', tld, language, {
        city: cityId,
        aeroex: 1,
    });
}
