import stableStringify from 'json-stable-stringify';

export function validateSize(size) {
    if (typeof size !== 'number' || size < -1 || size === 0) {
        throw new Error(
            'Параметр "size" должен быть положительным числом ' +
                'или быть равным -1',
        );
    }

    return true;
}

export function validateLifetime(lifetime) {
    if (typeof lifetime !== 'number' || lifetime < -1 || lifetime === 0) {
        throw new Error(
            'Параметр "lifetime" должен быть положительным числом ' +
                'или быть равным -1',
        );
    }

    return true;
}

/**
 * Походит для кэширования функций. Имеет сложность O(1).
 * Использовать лучше в тех случаях, когда объекты params небольшие, а количество
 * значений в кеше большое.
 */
export class CacheByParamsInMap {
    constructor({size, lifetime}) {
        validateSize(size);
        validateLifetime(lifetime);

        this.size = size;
        this.lifetime = lifetime;
        this.data = new Map();
        this.dataStringify = new WeakMap();
    }

    stringify(params) {
        let result = this.dataStringify.get(params);

        if (result) {
            return result;
        }

        result = stableStringify(params);

        // кэшируем нормализованный JSON, чтобы снизить оверхэд при последовательных
        // вызовах get/set во внешнем коде
        this.dataStringify.set(params, result);

        return result;
    }

    get(params) {
        const stringParams = this.stringify(params);

        return this.data.get(stringParams);
    }

    set(params, value) {
        const key = this.stringify(params);

        this.data.set(key, value);

        if (this.size !== -1 && this.data.size > this.size) {
            const iterator = this.data.entries();

            while (this.data.size > this.size) {
                const {value: valueArray, done} = iterator.next();

                if (done) {
                    break;
                }

                this.data.delete(valueArray[0]);
            }
        }

        if (this.lifetime !== -1) {
            setTimeout(() => this.data.delete(key), this.lifetime);
        }
    }

    del(params) {
        const key = this.stringify(params);

        this.data.delete(key);
    }
}
