import isServerSide from '../isServerSide';

type PrimitiveMap = Map<PrimitiveType, any>;
type ObjectMap = WeakMap<object, any>;

interface IWeakCacheNode {
    set(key: any, value: any): this;
    get(key: any): any;

    setResult(result: any): this;
    getResult(): any;
}

class WeakCacheNode implements IWeakCacheNode {
    protected primitiveMap?: PrimitiveMap;
    protected objectMap?: ObjectMap;
    protected result;

    protected getPrimitiveMap(): PrimitiveMap {
        if (!this.primitiveMap) {
            this.primitiveMap = new Map<PrimitiveType, any>();
        }

        return this.primitiveMap;
    }

    protected getObjectMap(): ObjectMap {
        if (!this.objectMap) {
            this.objectMap = new WeakMap<object, any>();
        }

        return this.objectMap;
    }

    protected getMap(key: PrimitiveType): PrimitiveMap;
    protected getMap(key: object): ObjectMap;
    protected getMap(key: any): PrimitiveMap | ObjectMap {
        const type = typeof key;

        return (type === 'object' && key !== null) || type === 'function'
            ? this.getObjectMap()
            : this.getPrimitiveMap();
    }

    set(key: any, value): this {
        const map = this.getMap(key);

        map.set(key, value);

        return this;
    }

    get(key): any {
        const map = this.getMap(key);

        return map && map.get(key);
    }

    setResult(result: any): any {
        this.result = result;

        return this;
    }

    getResult(): any {
        return this.result;
    }
}

interface IWeakCache {
    set(args: any[], result: any): this;
    get(args: any[]): any;
}

class WeakCache implements IWeakCache {
    protected root = new WeakCacheNode();

    set(args: any[], result: any): this {
        let item = this.root;
        let i = 0;
        const l = args.length;

        for (; i < l; i++) {
            const arg = args[i];
            let next = item.get(arg);

            if (!next) {
                next = new WeakCacheNode();
                item.set(arg, next);
            }

            item = next;
        }

        item.setResult(result);

        return this;
    }

    get(args: any[]): any {
        let item = this.root;
        const l = args.length;

        for (let i = 0; i < l; i++) {
            item = item.get(args[i]);

            if (!item) {
                return;
            }
        }

        return item.getResult();
    }
}

// Позволяет кешировать результат работы функции
export function makeCacheable<T extends any[], R extends any>(
    func: (...args: T) => R,
): (...args: T) => R {
    if (isServerSide()) {
        // Заплатка - будем чинить вот тут: https://st.yandex-team.ru/RASPFRONT-8561
        return func;
    }

    const cache = new WeakCache();

    return function (...args) {
        let result = cache.get(args);

        if (result === undefined) {
            result = func(...args);
            cache.set(args, result);
        }

        return result;
    };
}

/**
 * Позволяет кешировать результат работы функции,
 * принимающие в качестве параметра объект,
 * который впоследствии деструктуризируется
 *
 * Внимание!
 * Функция подходит только для кеширования функций с постоянным числом аргументов.
 * Т.е. если вы передаете в функцию, которую возвращает данная функция (закешированная функция),
 * объект с одним свойством { foo: true }, а потом снова вызываете ее с аргументом
 * { bar: true }, то вернется закешированное значение первого вызова, а это может быть совсем
 * не то что вы ожидали.
 * Данная функция будет выпилена вот здесь: https://st.yandex-team.ru/RASPFRONT-7159
 *
 * @param {Function} func
 * @return {function(*=): *}
 */
export function makeCacheableDestructure<T extends object, R extends any>(
    func: (arg: T) => R,
): (arg: T) => R {
    if (isServerSide()) {
        // Заплатка - будем чинить вот тут: https://st.yandex-team.ru/RASPFRONT-8561
        return func;
    }

    const cache = new WeakCache();

    return function (objectArgs) {
        const args = Object.keys(objectArgs)
            .sort()
            .map(key => objectArgs[key]);
        let result = cache.get(args);

        if (result === undefined) {
            result = func(objectArgs);
            cache.set(args, result);
        }

        return result;
    };
}

export const bindWithCache = makeCacheable((fn, ctx, ...args) =>
    typeof fn === 'function' ? fn.bind(ctx, ...args) : null,
);
