import {
    TIME_UNITS_LIST,
    TIME_UNITS_LENGTHS,
    DEFAULT_DURATION_FORMAT_OPTIONS,
} from './constants';
import {CHAR_DASH} from '../stringUtils';

import {EDurationShortening} from '../../interfaces/date/EDurationShortening';

import {roundDuration} from './durations';

import keyset from '../../i18n/time';

export const HUMAN = 'D MMMM';
export const TIME = 'HH:mm';
export const HOURS = 'H';
export const YEAR = 'YYYY';
export const RU_LOCALE_DATE = 'DD.MM.YYYY';
export const SHORT_DATE = 'DD.MM';
export const HUMAN_SHORT = 'D MMM';
export const ROBOT = 'YYYY-MM-DD';
export const HUMAN_MONTH = 'MMMM';
export const ROBOT_MONTH = 'YYYY-MM';
export const HUMAN_WEEKDAY = 'dddd';
export const HUMAN_WEEKDAY_SHORT = 'dd';
export const ROBOT_DATETIME = 'YYYY-MM-DD HH:mm:ss';
export const HUMAN_MONTH_WITH_YEAR = 'MMMM YYYY';
export const HUMAN_DATE_WITH_YEAR = 'D MMMM YYYY';
export const HUMAN_DATE_WITH_WEEKDAY = `${HUMAN}, ${HUMAN_WEEKDAY}`;
export const HUMAN_DATE_SHORT_WITH_SHORT_WEEKDAY = `${HUMAN_SHORT}, ${HUMAN_WEEKDAY_SHORT}`;
export const MOMENT = 'YYYY-MM-DDTHH:mm:ssZ';
export const MOMENT_WITHOUT_TIMEZONE = 'YYYY-MM-DDTHH:mm:ss';

const fullKey = key => key;
const shortKey = key => `${key}-short`;

export function formatDuration(duration, opts) {
    const {shortestUnit, longestUnit, short} = {
        ...DEFAULT_DURATION_FORMAT_OPTIONS,
        ...opts,
    };

    let index = TIME_UNITS_LIST.indexOf(longestUnit);
    const lastIndex = TIME_UNITS_LIST.indexOf(shortestUnit);
    const units = [];
    let rest = duration.asMilliseconds();

    while (index >= lastIndex && rest > 0) {
        const unit = TIME_UNITS_LIST[index];
        const unitLength = TIME_UNITS_LENGTHS[unit];
        const unitCount = Math.floor(rest / unitLength);

        if (unitCount > 0) {
            units.push([unit, unitCount]);
            rest -= unitCount * unitLength;
        }

        index--;
    }

    const getKey =
        short === EDurationShortening.NEVER
            ? fullKey
            : short === EDurationShortening.ALWAYS
            ? shortKey
            : units.length > 1
            ? shortKey
            : fullKey;

    return units
        .map(([unit, count]) => keyset(getKey(unit), {count}))
        .join(' ');
}

export function humanizeDuration(duration, opts) {
    const roundedDuration = roundDuration(duration, opts);

    return formatDuration(roundedDuration, opts);
}

/**
 * Форматирует разницу во времени (например: 390 => "+6:30")
 * @param {number} timeDifference - разница в минутах
 * @return {string}
 */
export function formatTimeDifference(timeDifference) {
    const hours = Math.round(timeDifference / 60);
    const minutes = timeDifference % 60;

    return `${timeDifference >= 0 ? '+' : CHAR_DASH}${Math.abs(hours)}${
        minutes ? `:${minutes}` : ''
    }`;
}
