import {Moment} from 'moment';

import {HUMAN_SHORT, TIME} from './formats';
import {CHAR_NBSP, CHAR_LIST_MARKER} from '../stringUtils';

import {isSameTime} from './utils';

import timezoneKeyset from '../../i18n/timezones';

function getRailwayTimeCode(railwayTimezone?: string): string {
    if (!railwayTimezone) {
        return 'local-time';
    }

    return railwayTimezone
        .substr(railwayTimezone.indexOf('/') + 1)
        .toLowerCase();
}

interface IGetRailwayTime {
    isAllDaysSearch: boolean;
    timeMoment: Moment;

    railwayMoment?: Moment;
    railwayTimezone?: string;
}

// Возвращает объект со временем и именем железнодорожной таймзоны
export default function getRailwayTime({
    isAllDaysSearch,
    timeMoment,
    railwayMoment,
    railwayTimezone,
}: IGetRailwayTime): {time: string; title: string} | null {
    if (!railwayMoment) {
        return null;
    }

    if (isSameTime(timeMoment, railwayMoment)) {
        return {
            time: timezoneKeyset('local-time-short'),
            title: timezoneKeyset('local-time'),
        };
    }

    const railwayTimeCode = getRailwayTimeCode(railwayTimezone);

    if (
        !timezoneKeyset.hasKey(railwayTimeCode) ||
        !timezoneKeyset.hasKey(`${railwayTimeCode}-short`)
    ) {
        return null;
    }

    const shortTimezoneTitle = timezoneKeyset(`${railwayTimeCode}-short`);
    const fullTimezoneTitle = timezoneKeyset(railwayTimeCode);

    let time = `${shortTimezoneTitle} ${railwayMoment.format(TIME)}`;
    const showDate =
        !isAllDaysSearch && timeMoment.date() !== railwayMoment.date();

    if (showDate) {
        time = `${time} ${CHAR_LIST_MARKER} ${railwayMoment
            .format(HUMAN_SHORT)
            .replace(/ /g, CHAR_NBSP)}`;
    }

    return {
        time,
        title: fullTimezoneTitle,
    };
}
