import {momentTimezone as moment} from '../../../reexports';

import {rangeToArray} from './utils';
import {makeCacheable, makeCacheableDestructure} from '../cache';

export const EMPTY_DAY = 0;

export function dayMatchesMask(day, mask, skipKeys = []) {
    const yearMask = mask[day.year()];

    if (!yearMask) {
        return false;
    }

    const monthMask = yearMask[day.month() + 1];

    if (!monthMask) {
        return false;
    }

    const filteredKeys = [...skipKeys, EMPTY_DAY];

    return !filteredKeys.includes(monthMask[day.date() - 1]);
}

export function getMaskValue(day, mask) {
    const yearMask = mask[day.year()];
    const monthMask = yearMask && yearMask[day.month() + 1];

    return monthMask ? monthMask[day.date() - 1] : null;
}

export function getRangeByMask({mask, timezone, utcOffset}) {
    if (typeof timezone === 'undefined' && typeof utcOffset === 'undefined') {
        throw new Error('Должен быть указан либо timezone либо utcOffset');
    }

    const years = Object.keys(mask);

    if (years.length === 0) {
        return null;
    }

    const minYear = Math.min(...years);
    const maxYear = Math.max(...years);

    const minMonth = Math.min(...Object.keys(mask[minYear]));
    const maxMonth = Math.max(...Object.keys(mask[maxYear]));

    let start;
    let end;

    if (typeof timezone !== 'undefined') {
        start = moment.tz([minYear, minMonth - 1], timezone);
        end = moment.tz([maxYear, maxMonth - 1], timezone).endOf('month');
    }

    if (typeof utcOffset !== 'undefined') {
        start = moment([minYear, minMonth - 1])
            .utcOffset(utcOffset, true)
            .startOf('month');
        end = moment([maxYear, maxMonth - 1])
            .utcOffset(utcOffset, true)
            .endOf('month');
    }

    return {
        start,
        end,
    };
}

export const getMonthsByMask = makeCacheableDestructure(
    ({mask, timezone, utcOffset}) => {
        if (
            typeof timezone === 'undefined' &&
            typeof utcOffset === 'undefined'
        ) {
            throw new Error('Должен быть указан либо timezone либо utcOffset');
        }

        const range = getRangeByMask({mask, timezone, utcOffset});

        return rangeToArray(range, 'months');
    },
);

function mergeMonths(monthA, monthB) {
    if (!monthA) {
        return monthB;
    }

    if (!monthB) {
        return monthA;
    }

    return monthA.map((day, i) => day || monthB[i]);
}

function merge(cb, objA, objB) {
    if (!objA) {
        return objB;
    }

    if (!objB) {
        return objA;
    }

    return Object.keys(objA).reduce(
        (result, key) => {
            result[key] = cb(objA[key], objB[key]);

            return result;
        },
        {...objB},
    );
}

const mergeFunction = merge.bind(null, merge.bind(null, mergeMonths));

function mergeMasksArray(masks) {
    return masks.reduce(mergeFunction, {});
}

export const getCommonMask = makeCacheable(segments => {
    const masks = [];

    segments.forEach(segment => {
        if (segment.runDays) {
            masks.push(segment.runDays);
        }

        if (segment.subSegments) {
            segment.subSegments.forEach(subsegment => {
                if (subsegment.runDays) {
                    masks.push(subsegment.runDays);
                }
            });
        }
    });

    return mergeMasksArray(masks);
});

export function replaceMaskValue(mask, value) {
    return Object.keys(mask).reduce((yearLevel, year) => {
        yearLevel[year] = Object.keys(mask[year]).reduce(
            (monthLevel, month) => {
                monthLevel[month] = mask[year][month].map(dateMask =>
                    dateMask !== EMPTY_DAY ? value : dateMask,
                );

                return monthLevel;
            },
            {},
        );

        return yearLevel;
    }, {});
}

/**
 * Возвращает массив дат активных дней согласно маске
 * @param {Object} mask - маска дней хождения
 * @param {string} [timezone] - таймзона
 * @param {Array} [skipKeys] - ключи, которые не должны учитываться как активные в маске
 * @param {string|number} [utcOffset] - смещение относительно UTC
 * @return {Array}
 */
export const getActivePlainMask = makeCacheableDestructure(
    ({mask, timezone, utcOffset, skipKeys = []}) => {
        if (
            typeof timezone === 'undefined' &&
            typeof utcOffset === 'undefined'
        ) {
            throw new Error('Должен быть указан либо timezone либо utcOffset');
        }

        const range = getRangeByMask({mask, timezone, utcOffset});

        return range
            ? rangeToArray({start: range.start, end: range.end}, 'day').filter(
                  day => dayMatchesMask(day, mask, skipKeys),
              )
            : [];
    },
);
