import moment, {Moment} from 'moment';

import Lang from '../../interfaces/Lang';

import getClosestMoment from './getClosestMoment';
import {MONTHS_NAMES} from './values';
import {startsWith} from '../stringUtils';

export default function parseDate(
    text: string,
    now: Moment,
    language: Lang,
): Moment | undefined {
    // Ищем что-нибудь похожее на 2016-15-01
    const robotMatch = /^\s*((\d{4})(-)(\d{1,2})(-)(\d{1,2}))$/.exec(text);
    // Ищем что-нибудь похожее на 1, 01-01, 01 01, 01.01, 01/01, 1 января, 1 января 2018
    const humanMatch =
        /^\s*(\d{1,2})(\s*-\s*|\s*\.\s*|\s*\/\s*|\s*)(\S*)(\2(\d{4}))?$/.exec(
            text,
        );

    // Не нашли, возвращаем как было
    if (!humanMatch && !robotMatch) {
        return;
    }

    let year: string | undefined;
    let date = '';
    let separator = '';
    let month = '';

    if (robotMatch) {
        year = robotMatch[2];
        date = robotMatch[6];
        separator = robotMatch[3];
        month = robotMatch[4];
    } else if (humanMatch) {
        year = humanMatch[5];
        date = humanMatch[1];
        separator = humanMatch[2];
        month = humanMatch[3];
    }

    if (date && !month && !year) {
        return getClosestMoment(date, now, !separator);
    }

    if (!month) {
        return;
    }

    const expectMonthAsNumber = separator.trim() !== '';
    const monthNumber = expectMonthAsNumber
        ? parseInt(month, 10) - 1
        : getNumberMonthByString(month, language);

    if (monthNumber === undefined || monthNumber < 0 || monthNumber > 11) {
        return;
    }

    if (year) {
        const parsedYear = moment(
            `${year}-${monthNumber + 1}-${date}`,
            'YYYY-M-D',
        );

        return parsedYear.isValid() ? parsedYear : undefined;
    }

    const parsedWithCurrentYear = moment(
        `${now.year()}-${monthNumber + 1}-${date}`,
        'YYYY-M-D',
    );

    if (
        parsedWithCurrentYear.isValid() &&
        parsedWithCurrentYear.isAfter(now.clone().subtract(1, 'month'))
    ) {
        return parsedWithCurrentYear;
    }

    const parsedWithNextYear = moment(
        `${now.year() + 1}-${monthNumber + 1}-${date}`,
        'YYYY-M-D',
    );

    if (parsedWithNextYear.isValid()) {
        return parsedWithNextYear;
    }

    return;
}

function getNumberMonthByString(
    monthString,
    language: Lang,
): number | undefined {
    const months: string[] = MONTHS_NAMES[language];

    return months && months.findIndex(month => startsWith(month, monthString));
}
