import {Moment} from '../../../reexports/moment-timezone';

import {HUMAN} from './formats';

import IMaskDaysOfWalking from '../../interfaces/maskDaysOfWalking/IMaskDaysOfWalking';

import {getActivePlainMask} from './mask';
import {isDateInRange} from './utils';

import maskPatternsKeyset from '../../i18n/mask-patterns';

const DATES_IN_STRING = 5;
const EVERYDAYS_LIMIT = 60;
const PERIOD_LIMIT = 7;

const EVERYDAY = 'EVERYDAY';
const PERIOD = 'PERIOD';
const CUSTOM = 'CUSTOM';

export function getCustomText(
    mask: IMaskDaysOfWalking,
    today: Moment,
    timezone: string,
): string {
    const activePlainMask = getActivePlainMask({mask, timezone});
    const end = activePlainMask[activePlainMask.length - 1];

    if (activePlainMask.length) {
        if (end.isSameOrAfter(today)) {
            let start = 0;

            while (
                activePlainMask[start] &&
                activePlainMask[start].isBefore(today, 'day')
            ) {
                start++;
            }

            const dates = activePlainMask.slice(start, start + DATES_IN_STRING);
            const formattedDates = dates.map((date, index) =>
                index === dates.length - 1 ||
                dates[index + 1].month() !== date.month()
                    ? date.format(HUMAN)
                    : date.date(),
            );

            return activePlainMask.length - start > DATES_IN_STRING
                ? maskPatternsKeyset('dates', {
                      dates: formattedDates.join(', '),
                  })
                : formattedDates.join(', ');
        }

        return maskPatternsKeyset('before-date', {date: end.format(HUMAN)});
    }

    return '';
}

export function getPeriodText(
    mask: IMaskDaysOfWalking,
    timezone: string,
): string {
    const activePlainMask = getActivePlainMask({mask, timezone});
    const start = activePlainMask.shift();
    const end = activePlainMask.pop();

    const dateFormat = maskPatternsKeyset(
        start.year() !== end.year() ? 'period-full-format' : 'period-format',
    );

    return maskPatternsKeyset('period-from-to', {
        start: start.format(dateFormat),
        end: end.format(dateFormat),
    });
}

export function getEverydayText(): string {
    return maskPatternsKeyset('everyday');
}

export function getMaskPattern(
    mask: IMaskDaysOfWalking,
    today: Moment,
    timezone: string,
): string | undefined {
    const activePlainMask = getActivePlainMask({mask, timezone});

    if (activePlainMask.length) {
        const start = activePlainMask[0];
        const end = activePlainMask[activePlainMask.length - 1];
        const duration = end.diff(start, 'day');

        if (
            duration === activePlainMask.length - 1 &&
            duration > PERIOD_LIMIT
        ) {
            if (
                isDateInRange(today, {start, end}) &&
                end.diff(today, 'day') > EVERYDAYS_LIMIT
            ) {
                return EVERYDAY;
            }

            return PERIOD;
        }

        return CUSTOM;
    }
}

export function maskToString(
    mask: IMaskDaysOfWalking,
    today: Moment,
    timezone: string,
): string {
    switch (getMaskPattern(mask, today, timezone)) {
        case EVERYDAY:
            return getEverydayText();
        case PERIOD:
            return getPeriodText(mask, timezone);

        default:
            return getCustomText(mask, today, timezone);
    }
}
