import {momentTimezone as moment} from '../../../reexports';

import {ROBOT, ROBOT_MONTH} from './formats';
import {TIMES_OF_DAY_LIST, TIMES_RANGES} from './values';

import {TransportType} from '../transportType';

import {makeCacheable} from '../cache';

export function getTimeOfDay(hour) {
    hour = Number(hour);

    return TIMES_OF_DAY_LIST.find(timeOfDay => {
        const timeRange = TIMES_RANGES[timeOfDay];

        const endTime =
            timeRange.start <= timeRange.end
                ? timeRange.end
                : timeRange.end + 24;

        return hour >= timeRange.start && hour < endTime;
    });
}

export function getTimesOfDay(startHour, endHour) {
    startHour = Number(startHour);
    endHour = Number(endHour);

    const startTimeOfDay = getTimeOfDay(startHour);
    const endTimeOfDay = getTimeOfDay(endHour);
    const allTimes = startTimeOfDay === endTimeOfDay && startHour > endHour;
    const singleTime = startTimeOfDay === endTimeOfDay && startHour < endHour;

    const times = [...TIMES_OF_DAY_LIST, ...TIMES_OF_DAY_LIST];
    const timesOfDay = [startTimeOfDay];
    let index = times.indexOf(startTimeOfDay);

    if (!singleTime) {
        do {
            index++;
            timesOfDay.push(times[index]);
        } while (times[index] !== endTimeOfDay);
    }

    return allTimes ? timesOfDay.slice(0, 4) : timesOfDay;
}

/**
 * Вернёт список времён суток соответствующий времени хождения заданного сегмента
 * @param {Object} segment - данные сегмента
 * @return {string[]}
 */
export function getIntervalTimesOfDay(segment) {
    const {thread, isInterval} = segment;

    if (isInterval && thread) {
        const {beginTime = '', endTime = ''} = thread;
        const beginHour = beginTime.split(':').shift();
        const endHour = endTime.split(':').shift();

        return getTimesOfDay(beginHour, endHour);
    }

    return [];
}

export function isSameTime(timeA, timeB) {
    return (
        timeA.hours() === timeB.hours() && timeA.minutes() === timeB.minutes()
    );
}

export function isSameISODateString(dateA, dateB) {
    const momentA = moment.tz(dateA, moment.ISO_8601, 'UTC');

    if (!momentA.isValid()) {
        return false;
    }

    const momentB = moment.tz(dateB, moment.ISO_8601, 'UTC');

    if (!momentB.isValid()) {
        return false;
    }

    return momentA.valueOf() === momentB.valueOf();
}

export function getNow() {
    let now = Date.now();

    if (typeof window !== 'undefined' && window.timeCorrection) {
        now += window.timeCorrection;
    }

    return now;
}

export const getMomentDate = makeCacheable((time, timezone) =>
    moment.tz(time, timezone),
);

export const getToday = makeCacheable(time => {
    const now =
        time instanceof moment
            ? time.startOf('day')
            : moment.tz(time.now, time.timezone);

    return now.startOf('day');
});

export const getCurrentDay = timezone => moment().tz(timezone);

export const getNextDay = timezone => getCurrentDay(timezone).add(1, 'days');

export function getLastYearPeriod({now, timezone}) {
    const year = moment.tz(now, timezone).year();

    return [year - 1, year].join('/');
}

export function getClosestWeekday(weekday, date) {
    const result = date.clone().weekday(weekday);

    if (result.isBefore(date)) {
        result.add(1, 'week');
    }

    return result;
}

const defaultRangeEnd = date => date.clone().add(11, 'month');

export const getRange = makeCacheable(date => ({
    start: date.clone().subtract(1, 'month').add(1, 'day'),
    end: defaultRangeEnd(date),
}));

export const getRangeFromDate = makeCacheable(date => ({
    start: date.clone(),
    end: date.clone().add(12, 'month'),
}));

export const getOrderRange = makeCacheable(
    (date, transportType, limit, isCppk) => {
        if (isCppk) {
            return {
                start: date.clone(),
                end: date.clone().add(limit, 'days'),
            };
        }

        switch (transportType) {
            case TransportType.plane:
                return {
                    start: date.clone(),
                    end: defaultRangeEnd(date),
                };
            case TransportType.train:
                return {
                    start: date.clone(),
                    end: date.clone().add(limit, 'days'),
                };

            default:
                return getRange(date);
        }
    },
);

export function isDateInRange(date, {start, end}) {
    return !date.isBefore(start) && !date.isAfter(end);
}

export function fitInRange(date, {start, end}) {
    if (date.isBefore(start)) {
        return date.clone().add(1, 'year');
    }

    if (date.isAfter(end)) {
        return date.clone().subtract(1, 'year');
    }

    return date;
}

export function isOutOfRange(date, {start, end}) {
    return date.isBefore(start) || date.isAfter(end);
}

export function getClosestMoment(dateStr, today, canAddDigit) {
    let date = parseInt(dateStr, 10);

    if (date > 31 || (date < 0 && !canAddDigit) || date < 1) {
        return;
    }

    const result = today.clone();
    const todayDate = today.date();
    const daysInCurrentMonth = today.daysInMonth();

    // Если в текущий месяц результат не вписывается, то точно впишется в следующий
    if (date > daysInCurrentMonth) {
        result.add(1, 'month').date(date);

        return result;
    }

    if (date < todayDate) {
        const dateWithZero = date * 10;

        if (
            canAddDigit &&
            dateWithZero <= daysInCurrentMonth &&
            dateWithZero + 9 >= todayDate
        ) {
            date = Math.max(todayDate, dateWithZero);
        } else {
            result.add(1, 'month');

            if (date > result.daysInMonth()) {
                result.add(1, 'month');
            }
        }
    }

    result.date(date);

    return result;
}

export function isWeekend(weekday) {
    return weekday === 5 || weekday === 6;
}

export function getParseParams({time, language, tld}) {
    return {time, language, tld};
}

export function getNextDayFormatted(date, timezone) {
    return moment.tz(date, timezone).add(1, 'days').format(ROBOT);
}

export function rangeToArray({start, end}, step) {
    const result = [];

    for (
        let item = start.clone().startOf(step);
        item.isBefore(end);
        item = item.clone().add(1, step)
    ) {
        result.push(item);
    }

    return result;
}

export function extendRange({start, end}, extendTo) {
    return {
        start: start.clone().startOf(extendTo),
        end: end.clone().endOf(extendTo),
    };
}

export const extendRangeToArray = makeCacheable((range, step, extendTo) => {
    range = extendRange(range, extendTo);

    return rangeToArray(range, step, extendTo);
});

export const getInactiveMonths = makeCacheable((allDates, activeDates) => {
    const activeMonths = activeDates.reduce(
        (result, date) => ({
            ...result,
            [date.format(ROBOT_MONTH)]: true,
        }),
        {},
    );

    return allDates.reduce((result, date) => {
        const formatedDate = date.format(ROBOT_MONTH);

        return {
            ...result,
            [formatedDate]: !activeMonths[formatedDate],
        };
    }, {});
});

/**
 * Возвращает объект с указанием оставшегося времени
 * @param {Moment} firstMoment - moment объект
 * @param {Moment} secondMoment - moment объект
 * @return {Object} - { hours: Number, minutes: Number }
 */
export const getRemainingTime = (firstMoment, secondMoment) => {
    if (firstMoment.isBefore(secondMoment)) {
        const remainingTime = secondMoment.diff(firstMoment);
        const minute = 1000 * 60;
        const hour = 60 * minute;

        return {
            hours: Math.floor(remainingTime / hour),
            minutes: Math.floor((remainingTime % hour) / minute),
        };
    }

    return {
        hours: 0,
        minutes: 0,
    };
};

/**
 * Время хождения интервальных рейсов имеет смысл отображать
 * только в местном времени, поэтому просто выводим как есть (только отрезаем секунды)
 * @param {string} timeStr - строка со временем
 * @return {string}
 */
export function formatLocalTime(timeStr) {
    return timeStr.substr(0, 5);
}

/**
 * Вернёт дату со временем установленным на начало дня
 * Нормальные люди используют startOfDay, но при использовании таймзон и вычислений с ним возникают проблемы:
 * https://github.com/moment/moment/issues/3774
 * @param { Moment } momentDate - moment объект
 * @return { Moment }
 */
export function getStartOfDate(momentDate) {
    return momentDate.clone().hours(0).minutes(0).seconds(0).milliseconds(0);
}
