import {momentTimezone as moment} from '../../../reexports';

import {HUMAN, HUMAN_WEEKDAY, HUMAN_WEEKDAY_SHORT} from './formats';

import Tld from '../../interfaces/Tld';
import Lang from '../../interfaces/Lang';
import TimeOfDay from '../../interfaces/date/TimeOfDay';
import DateSpecialValue from '../../interfaces/date/DateSpecialValue';
import Keyset from '../../interfaces/Keyset';

import ruSpecialValues from '../../i18n/bt-lib-datepicker-values/ru';
import ukSpecialValues from '../../i18n/bt-lib-datepicker-values/uk';

const locales = ['ru', 'uk'];

export const MIN_MONTHS_TO_FULL_CALENDAR = 3;

export const TIMES_OF_DAY_LIST = [
    TimeOfDay.morning,
    TimeOfDay.day,
    TimeOfDay.evening,
    TimeOfDay.night,
];

export const TIMES_RANGES = {
    [TimeOfDay.night]: {
        start: 0,
        end: 6,
    },
    [TimeOfDay.morning]: {
        start: 6,
        end: 12,
    },
    [TimeOfDay.day]: {
        start: 12,
        end: 18,
    },
    [TimeOfDay.evening]: {
        start: 18,
        end: 0,
    },
};

export const SPECIAL_VALUES_PRIORITY = [
    DateSpecialValue.allDays,
    DateSpecialValue.today,
    DateSpecialValue.tomorrow,
    DateSpecialValue.afterTomorrow,
    DateSpecialValue.yesterday,
    DateSpecialValue.inAWeek,
    DateSpecialValue.inAMonth,
];

export const SPECIAL_VALUES_OFFSETS = {
    today: {days: 0},
    tomorrow: {days: 1},
    'after-tomorrow': {days: 2},
    yesterday: {days: -1},
    'in-a-week': {weeks: 1},
    'in-a-month': {months: 1},
};

const pluckFromMoment = (
    method: string,
    count: number,
    format: string,
): Record<string, string[]> =>
    locales.reduce((result, locale) => {
        result[locale] = [];

        const m = moment().locale(locale);

        for (let i = 0; i < count; i++) {
            m[method](i);
            result[locale].push(
                m.format(format).toLowerCase().split(' ').pop(),
            );
        }

        return result;
    }, {});

const pluckFromI18n = (
    keysets: Record<string, Keyset>,
): Record<string, Record<string, string>> =>
    locales.reduce((result, locale) => {
        result[locale] = SPECIAL_VALUES_PRIORITY.reduce((values, key) => {
            values[key] = keysets[locale](key);

            return values;
        }, {});

        return result;
    }, {});

export const SPECIAL_VALUES = pluckFromI18n({
    ru: ruSpecialValues,
    uk: ukSpecialValues,
});
export const MONTHS_NAMES = pluckFromMoment('month', 12, HUMAN);
export const WEEKDAY_NUMBERS = [0, 1, 2, 3, 4, 5, 6];
export const WEEKDAY_NAMES = pluckFromMoment('weekday', 7, HUMAN_WEEKDAY);
export const WEEKDAY_NAMES_SHORT = pluckFromMoment(
    'weekday',
    7,
    HUMAN_WEEKDAY_SHORT,
);

export const MAIN_SPECIAL_KEYS = [
    DateSpecialValue.today,
    DateSpecialValue.tomorrow,
    DateSpecialValue.allDays,
];

const LANGUAGE_PRIORITIES_BY_TLD = {
    ua: {
        ru: ['ru', 'uk'],
        uk: ['uk', 'ru'],
    },
};

export const getLanguagePriority = (tld: Tld, language: Lang): string[] =>
    (LANGUAGE_PRIORITIES_BY_TLD[tld] &&
        LANGUAGE_PRIORITIES_BY_TLD[tld][language]) || ['ru'];

export function getTodayPhrase(lang = 'ru'): string {
    const keysets = {
        ru: ruSpecialValues,
        uk: ukSpecialValues,
    };

    return keysets[lang]('today');
}
