import applyFilters from './utils/applyFilters';

export default {
    type: 'unknown', // название фильтра

    skipIfReset: false, // Если установлено в true, то при сбросе фильтра данный параметр не сбросится

    getDefaultValue() {
        return [];
    },

    getDefaultOptions() {
        return [];
    },

    /**
     * Определяет доступен ли данный фильтр для контекста запроса
     * (направление, тип транспорта, дата и т.д. search.context в state)
     * @return {boolean}
     */
    isAvailableForContext() {
        return true;
    },

    /**
     * Проверяет доступность данного фильтра для переданных доступных значений
     * @param {Object|Array} options
     * @return {boolean}
     */
    isAvailableWithActiveOptions(options) {
        return this.isAvailableWithOptions(options);
    },

    isAvailableWithOptions() {
        return true;
    },

    /**
     * Сериализация данных о фильтре в простой объект,
     * который позже преобразуется в get-параметры url
     * @return {Object}
     */
    serializeToQuery() {
        return {};
    },

    isDefaultValue(value) {
        return value.length === 0;
    },

    /**
     * Проверяет должен ли применяться данный фильтр
     * @param {*} value - текущее значение фильтра
     * @return {boolean}
     */
    shouldApply(value) {
        return !this.isDefaultValue(value);
    },

    /**
     * Возвращает список доступных значений данного фильтра
     * @param {Array} segments
     * @return {Array}
     */
    getOptions(segments) {
        return segments.reduce(
            (prevOptions, segment) => this.updateOptions(prevOptions, segment),
            this.getDefaultOptions(),
        );
    },

    /**
     * Возвращает начальное состояние данного фильтра
     * @param {Array} segments
     * @return {{value: (*|boolean), options: *, activeOptions: *, availableWithOptions: (*|boolean), availableWithActiveOptions: (*|boolean), type: *, filteredSegmentIndices: any[]}}
     */
    initFilterData(segments) {
        const options = this.getOptions(segments);
        const availableWithOptions = this.isAvailableWithOptions(options);
        const filteredSegmentIndices = new Array(segments.length).fill(true);

        return {
            value: this.getDefaultValue(),
            options,
            activeOptions: options,
            availableWithOptions,
            availableWithActiveOptions: availableWithOptions,
            type: this.type,
            filteredSegmentIndices,
        };
    },

    /**
     * Возвращает массив, в котором ключом является индекс сегмента, а значение булевый признак,
     * того, что сегмент прошел фильтрацию по данному признаку
     * @param {*} value - значение фильтра
     * @param {Array} segments - сегменты
     * @return {Array}
     */
    getFilteredSegmentIndices(value, segments) {
        if (!this.shouldApply(value)) {
            return new Array(segments.length).fill(true);
        }

        return segments.map(segment => this.apply(value, segment));
    },

    /**
     * Возвращает валидное значения фильтра
     * @param {*} value - текущее значение фильтра (из state)
     * @return {*}
     */
    validateValue(value) {
        return value;
    },

    setFilterValue({filtersData, value, segments}) {
        value = this.validateValue(value, filtersData[this.type].options);
        const newFiltersData = {...filtersData};

        newFiltersData[this.type] = {
            ...filtersData[this.type],
            value,
            filteredSegmentIndices: this.getFilteredSegmentIndices(
                value,
                segments,
            ),
        };

        return newFiltersData;
    },

    getActiveOptions({filtersData, segments}) {
        return segments.reduce((options, segment, segmentIndex) => {
            if (
                applyFilters({
                    segmentIndex,
                    filtersData,
                    excludeFilterType: this.type,
                })
            ) {
                return this.updateOptions(options, segment);
            }

            return options;
        }, this.getDefaultOptions());
    },

    /**
     * Возвращает обновленное стостояние доступных значений фильтров
     * @param {Array} segments
     * @param {Object} filtersData - состояние фильтров
     * @return {Object}
     */
    updateActiveOptions({segments, filtersData}) {
        const activeOptions = this.getActiveOptions({filtersData, segments});
        const filterData = {
            ...filtersData[this.type],
            activeOptions,
            availableWithActiveOptions:
                this.isAvailableWithActiveOptions(activeOptions),
        };

        return {
            ...filtersData,
            [this.type]: filterData,
        };
    },

    /**
     * Возвращает текст опции
     * @param {Object|string} option
     * @return {string}
     */
    getOptionText(option) {
        const {text} = option;

        return text;
    },
};
