import fromPairs from 'lodash/fromPairs';

import getFilterManagers from './filterManagerProvider';
import updateFiltering from './utils/updateFiltering';

export function getEffectiveManagers(filtering) {
    return getFilterManagers().filter(
        manager => filtering.filters[manager.type],
    );
}

export function getFilteringBySegments({
    context,
    segments,
    isSuburbanSearchResult,
}) {
    const managers = getFilterManagers();

    const filtersData = managers
        .filter(manager =>
            manager.isAvailableForContext(context, {isSuburbanSearchResult}),
        )
        .map(manager => [manager.type, manager.initFilterData(segments)]);

    return {
        filters: fromPairs(filtersData),
        filteredSegmentIndices: new Array(segments.length).fill(true),
    };
}

export function noSegmentsAllowedByFilters(filtering) {
    return !filtering.filteredSegmentIndices.filter(Boolean).length;
}

export function setFilterValue({filtering, filterType, value, segments}) {
    const managers = getEffectiveManagers(filtering);
    const newValues = {[filterType]: value};

    return updateFiltering({filtering, managers, newValues, segments});
}

export function updateFilteringByUrlQuery({filtering, query, segments}) {
    const managers = getEffectiveManagers(filtering);

    const newValues = fromPairs(
        managers.map(manager => [
            manager.type,
            manager.deserializeFromQuery(query),
        ]),
    );

    return updateFiltering({filtering, managers, newValues, segments});
}

export function getFilteringQuery(filtering) {
    const managers = getEffectiveManagers(filtering);

    return managers.reduce((query, manager) => {
        return {
            ...query,
            ...manager.serializeToQuery(filtering.filters[manager.type].value),
        };
    }, {});
}

export function filtersWereApplied(filtering) {
    const {filters = {}} = filtering;

    return !getEffectiveManagers(filtering).reduce(
        (result, manager) =>
            result &&
            (manager.skipIfReset ||
                manager.isDefaultValue(filters[manager.type].value)),
        true,
    );
}

export function resetFiltersToDefault(filtering, segments) {
    const managers = getEffectiveManagers(filtering);
    const {filters = {}} = filtering;
    const newValues = fromPairs(
        managers.map(manager => [
            manager.type,
            manager.skipIfReset
                ? filters[manager.type].value
                : manager.getDefaultValue(),
        ]),
    );

    return updateFiltering({filtering, managers, newValues, segments});
}

export function anyFiltersAvailable(filtering) {
    const filterKeys = Object.keys(filtering.filters);

    for (let i = 0; i < filterKeys.length; i++) {
        if (filtering.filters[filterKeys[i]].availableWithOptions) {
            return true;
        }
    }

    return false;
}
