import {FilterTransportType} from '../transportType';
import ISegment from '../../interfaces/segment/ISegment';
import ITransfer from '../../interfaces/transfer/ITransfer';
import ILastochkaOptions from '../../interfaces/state/search/Filters/ILastochkaOptions';

import baseManager from './baseFilterManager';
import isLastochka from '../segments/isLastochka';

export const LASTOCHKA_FILTER_TYPE = 'lastochka';

export default {
    ...baseManager,

    type: LASTOCHKA_FILTER_TYPE,

    getDefaultValue(): boolean {
        return false;
    },

    getDefaultOptions(): ILastochkaOptions {
        return {withLastochka: false, withoutLastochka: false};
    },

    supportedFor: new Set([
        FilterTransportType.all,
        FilterTransportType.suburban,
    ]),

    apply(value: boolean, segment: ISegment | ITransfer): boolean {
        return !value || isLastochka(segment);
    },

    updateOptions(
        options: ILastochkaOptions,
        segment: ISegment | ITransfer,
    ): ILastochkaOptions {
        if (options.withLastochka && options.withoutLastochka) {
            return options;
        }

        if (isLastochka(segment)) {
            return {
                ...options,
                withLastochka: true,
            };
        }

        return {
            ...options,
            withoutLastochka: true,
        };
    },

    serializeToQuery(lastochkaOnly: boolean): Record<string, string> {
        return lastochkaOnly ? {lastochka: 'y'} : {};
    },

    deserializeFromQuery(query: Record<string, string>): boolean {
        const {lastochka} = query;

        return Boolean(lastochka && lastochka.toLowerCase() === 'y');
    },

    isDefaultValue(lastochkaOnly: boolean): boolean {
        return lastochkaOnly === this.getDefaultValue();
    },

    isAvailableForContext(context: {
        transportType: FilterTransportType;
    }): boolean {
        return this.supportedFor.has(context.transportType);
    },

    isAvailableWithOptions(options: ILastochkaOptions): boolean {
        return Boolean(options.withLastochka);
    },
};
