import isPlaneObject from 'lodash/isPlainObject';

import CurrencyCode from '../../interfaces/CurrencyCode';

import baseManager from './baseFilterManager';
import getSegmentPrices from './utils/getSegmentPrices';
import {
    getBaseTariffClassKeys,
    filterTariffClassKeysByTrainTariffClass,
} from '../segments/getBaseTariffClassKeys';

const countRanges = 4;
const frequencyRates = {
    [CurrencyCode.rub]: 100,
    [CurrencyCode.uah]: 50,
};

function floorToFrequencyRate(number, frequencyRate) {
    return Math.floor(number / frequencyRate) * frequencyRate;
}

function ceilToFrequencyRate(number, frequencyRate) {
    return Math.ceil(number / frequencyRate) * frequencyRate;
}

export default {
    ...baseManager,

    type: 'priceRange',

    apply(value, segment) {
        const segmentPrices = getSegmentPrices(segment).map(
            price => price.value,
        );

        return segmentPrices.some(price =>
            value.some(range => this.priceInRange(range, price)),
        );
    },

    /**
     * Проверяет входит ли цена в диапазон
     * @param {Object} range
     * @param {number} range.min
     * @param {number} range.max
     * @param {number} price
     * @return {boolean}
     */
    priceInRange(range, price) {
        return (
            this.priceCorrespondsToMin(price, range.min) &&
            this.priceCorrespondsToMax(price, range.max)
        );
    },

    /**
     * Проверяет соответствие цены нижней границе диапазона
     * @param {number} price
     * @param {number} min
     * @return {boolean}
     */
    priceCorrespondsToMin(price, min) {
        return price >= min;
    },

    /**
     * Проверяет соответствие цены верхней границе диапазона
     * @param {number} price
     * @param {number} max
     * @return {boolean}
     */
    priceCorrespondsToMax(price, max) {
        return price < max;
    },

    /**
     * Возвращает массив подходящих под фильтр типов вагонов
     * @param {Array} value - значение фильтра
     * @param {Object} segment - сегмент
     * @return {string[]} - массив тарифов вагонов сегмента
     */
    getSuitableSegmentTariffClasses(value, segment) {
        const prices = getSegmentPrices(segment);

        return prices
            .filter(priceObject =>
                value.some(range =>
                    this.priceInRange(range, priceObject.value),
                ),
            )
            .map(priceObject => priceObject.tariffName);
    },

    updateOptions(options) {
        return options;
    },

    serializeToQuery(value) {
        if (this.isDefaultValue(value)) {
            return {};
        }

        return {
            priceRange: value.map(val => val.value).join(','),
        };
    },

    deserializeFromQuery({priceRange}) {
        if (!priceRange || typeof priceRange !== 'string') {
            return this.getDefaultValue();
        }

        return priceRange
            .split(',')
            .map(value => value.trim())
            .map(this.parseOptionFromString)
            .filter(Boolean);
    },

    /**
     * Возвращает минимальный шаг для диапазона цен
     * @param {string} currency
     * @return {*|number}
     */
    getFrequencyRate(currency = '') {
        return frequencyRates[currency.toUpperCase()] || 100;
    },

    /**
     * Возвращает количество цен в сегментах и Map, в котором цены разложены по промежуткам,
     * кратным мнимальному шагу для диапазона (frequencyRate)
     * @param {Array} segments
     * @param {string} currency
     * @return {{totalCount: number, roundRanges: Map<any, any>}}
     */
    getRangesByFrequencyRate(segments, currency = '') {
        const roundRanges = new Map();
        let totalCount = 0;
        const frequencyRate = this.getFrequencyRate(currency);

        // раскладываем все цены на промежутки равные frequencyRate
        segments.forEach(segment =>
            getSegmentPrices(segment).forEach(priceObject => {
                const price = priceObject.value;
                let min = floorToFrequencyRate(price, frequencyRate);
                let max = ceilToFrequencyRate(price, frequencyRate);

                if (min === max) {
                    min -= frequencyRate;
                }

                if (price === max) {
                    // перекидываем цену в следующий промежуток
                    min += frequencyRate;
                    max += frequencyRate;
                }

                const rangeKey = `${min}-${max}`;
                const range = roundRanges.get(rangeKey) || {
                    min,
                    max,
                    count: 0,
                };

                range.count++;
                roundRanges.set(rangeKey, range);
                totalCount++;
            }),
        );

        return {
            totalCount,
            roundRanges,
        };
    },

    /**
     * Возвращает список доступных значений данного фильтра
     * @param {[]} segments
     * @return {[]}
     */
    getOptions(segments) {
        // определяем национальную валюту
        let currency = null;

        segments.some(segment => {
            const segmentPrices = getSegmentPrices(segment);

            if (segmentPrices.length) {
                currency = segmentPrices[0].currency;
            }

            return currency;
        });

        const {totalCount, roundRanges} = this.getRangesByFrequencyRate(
            segments,
            currency || '',
        );
        // собираем все промежутки в большие интервалы
        const approximateCountInRange = Math.floor(totalCount / countRanges);
        let ranges = [];
        let currentRange = {
            count: 0,
            min: null,
            max: null,
        };

        [...roundRanges.values()]
            .sort((a, b) => a.min - b.min)
            .forEach(roundRange => {
                if (currentRange.min === null) {
                    currentRange.min = roundRange.min;
                }

                const newCount = currentRange.count + roundRange.count;
                const rangeFilled =
                    currentRange.count && newCount >= approximateCountInRange;

                if (rangeFilled && ranges.length < countRanges - 1) {
                    ranges.push(currentRange);
                    const min = currentRange.max;

                    currentRange = {
                        count: 0,
                        min,
                        max: roundRange.max,
                    };
                }

                currentRange.count += roundRange.count;
                currentRange.max = roundRange.max;
            });
        ranges.push(currentRange);
        ranges = ranges.filter(range => range.count);

        return ranges.length > 1
            ? ranges.map(range => {
                  range.value = `${range.min}-${range.max}`;
                  range.currency = currency;

                  return range;
              })
            : [];
    },

    /**
     * Возвращает только те объекты с ценами, что будут показаны пользователю
     * @param {Object} segment
     * @param {Object} filtersData
     * @return {Array<{value: number, currency: (string|null), tariffName: string}>}
     */
    getSegmentPricesForDisplay({segment, filtersData}) {
        let prices = getSegmentPrices(segment);
        let tariffClassKeys = getBaseTariffClassKeys(segment);

        // список тарифов, подходящих под фильтр типа вагона
        tariffClassKeys = filterTariffClassKeysByTrainTariffClass(
            tariffClassKeys,
            segment,
            filtersData,
        );

        // список цен, подходящий под данные тарифы
        prices = prices.filter(priceObject =>
            tariffClassKeys.includes(priceObject.tariffName),
        );

        return prices;
    },

    /**
     * Возвращает список доступных значений для данного фильтра,
     * с учетом результатов других фильтров
     * @param {{}} filtersData - состояние фильтров
     * @param {[]} segments
     * @param {[]} filteredSegments
     * @return {*}
     */
    getActiveOptions({filtersData, segments, filteredSegments = null}) {
        filteredSegments =
            filteredSegments ||
            this.getFilteredSegments({segments, filtersData});
        const options = this.getOptions(segments);

        // возвращаем только те опции (дипазоны цен), что соответсвуют оставшимся сегементам
        return options.filter(option =>
            filteredSegments.some(segment => {
                const prices = this.getSegmentPricesForDisplay({
                    segment,
                    filtersData,
                });

                return prices.some(({value}) =>
                    this.priceInRange(option, value),
                );
            }),
        );
    },

    formatOptions(options, formatText = null) {
        formatText = formatText || (option => `${option.min} - ${option.max}`);

        return options.map((option, index) => ({
            value: option.value,
            text: formatText(option, index, options.length),
        }));
    },

    reformatOptions(formattedOptions, options) {
        return formattedOptions
            .map(formattedOption =>
                options.find(option => {
                    if (typeof formattedOption === 'string') {
                        return formattedOption === option.value;
                    }

                    if (isPlaneObject(formattedOption)) {
                        return (
                            formattedOption.min === option.min &&
                            formattedOption.max === option.max
                        );
                    }

                    return false;
                }),
            )
            .filter(Boolean);
    },

    formatValue(value) {
        return value.map(valueItem => valueItem.value).filter(Boolean);
    },

    validateValue(value, options) {
        if (!Array.isArray(value)) {
            return this.getDefaultValue();
        }

        return value
            .map(
                val =>
                    val &&
                    options.find(
                        option =>
                            typeof val.min !== 'undefined' &&
                            typeof val.max !== 'undefined' &&
                            val.min === option.min &&
                            val.max === option.max,
                    ),
            )
            .filter(Boolean);
    },

    parseOptionFromString(value) {
        const range = value
            .split('-')
            .map(str => parseInt(str.trim(), 10))
            .filter(number => !isNaN(number));

        if (range.length !== 2 || range[0] < 0 || range[1] <= 0) {
            return null;
        }

        return {
            min: range[0],
            max: range[1],
        };
    },

    /**
     * Обновляет список опций на основе данных об отфильтрованных сегментах
     * @param {Object} params
     * @param {Object} params.options
     * @param {Object} params.filtersData
     * @param {Array} params.segments
     * @param {Array} [params.filteredSegments]
     * @return {Array}
     */
    updateOptionsForFilteredSegments({
        options,
        filtersData,
        segments,
        filteredSegments,
    }) {
        filteredSegments =
            filteredSegments ||
            this.getFilteredSegments({segments, filtersData});
        const newOptions = [];

        // создаем копию option и обнуляем количество тарифов в диапазонах
        options.forEach(option => {
            newOptions.push({
                ...option,
                count: 0,
            });
        });

        // получаем массив со всеми ценами (которые будут показаны пользователю)
        // отфильтрованных сегментов
        const prices = filteredSegments.reduce(
            (pricesArray, segment) =>
                pricesArray.concat(
                    this.getSegmentPricesForDisplay({segment, filtersData}).map(
                        ({value}) => value,
                    ),
                ),
            [],
        );

        // обновляем информацию о количестве тарифов в диапазонах
        prices.forEach(price => {
            newOptions.some(option => {
                if (this.priceCorrespondsToMax(price, option.max)) {
                    option.count++;

                    return true;
                }

                return false;
            });
        });

        return newOptions;
    },
};
