import {
    ALL_TYPE,
    FILTERABLE_TRANSPORT_TYPES,
    getTransportTypeName,
} from '../transportType';
import baseManager from './baseFilterManager';

export default {
    ...baseManager,

    type: 'transport',

    apply(value, segment) {
        // Пересадочный сегмент должен содержать только выбранные в фильтре типы транспорта
        if (segment.isTransfer) {
            return segment.segments.every(transferSegment =>
                value.includes(transferSegment.transport.code),
            );
        }

        return value.includes(segment.transport.code);
    },

    updateOptions(options, segment) {
        let segmentTransportTypes;

        if (segment.isTransfer) {
            segmentTransportTypes = segment.segments.reduce(
                (transports, transferSegment) => {
                    if (!transports[transferSegment.transport.code]) {
                        transports[transferSegment.transport.code] = true;
                    }

                    return transports;
                },
                {},
            );
        } else {
            segmentTransportTypes = {[segment.transport.code]: true};
        }

        return FILTERABLE_TRANSPORT_TYPES.filter(
            transportType =>
                Boolean(segmentTransportTypes[transportType]) ||
                options.includes(transportType),
        );
    },

    serializeToQuery(value) {
        return {transport: value};
    },

    deserializeFromQuery({transport}) {
        if (!transport) {
            return this.getDefaultValue();
        }

        if (!Array.isArray(transport)) {
            transport = [transport];
        }

        return FILTERABLE_TRANSPORT_TYPES.filter(transportType =>
            transport.includes(transportType),
        );
    },

    isAvailableForContext(context) {
        return context.transportType === ALL_TYPE;
    },

    isAvailableWithOptions(options) {
        return options.length > 1;
    },

    validateValue(value = [], options = []) {
        return value.filter(item => options.includes(item));
    },

    getOptionText(option) {
        return getTransportTypeName(option);
    },
};
