import {CHAR_THINSP} from '../stringUtils';

import CurrencyCode from '../../interfaces/CurrencyCode';

import currencyAbbrKeyset from '../../i18n/currency-abbr';

const CURRENCY_SYMBOLS = {
    [CurrencyCode.rub]: '₽',
    [CurrencyCode.kzt]: '₸',
    [CurrencyCode.usd]: '$',
    [CurrencyCode.eur]: '€',
};

const BASIC_FORMATTERS = {
    EASTERN: (whole, cents) => (cents ? `${whole},${cents}` : whole),
    WESTERN: (whole, cents) => (cents ? `${whole}.${cents}` : whole),
};

const BASIC_FORMATTERS_WITH_SYMBOL = {
    EASTERN: (whole, cents, symbol) =>
        `${BASIC_FORMATTERS.EASTERN(whole, cents)}${CHAR_THINSP}${symbol}`,
    WESTERN: (whole, cents, symbol) =>
        `${symbol}${BASIC_FORMATTERS.WESTERN(whole, cents)}`,
};

const CURRENCY_FORMATTERS = {
    [CurrencyCode.rub]: BASIC_FORMATTERS.EASTERN,
    [CurrencyCode.uah]: BASIC_FORMATTERS.EASTERN,
    [CurrencyCode.kzt]: BASIC_FORMATTERS.EASTERN,
    [CurrencyCode.byn]: BASIC_FORMATTERS.EASTERN,
    [CurrencyCode.uzs]: BASIC_FORMATTERS.EASTERN,

    [CurrencyCode.usd]: BASIC_FORMATTERS.WESTERN,
    [CurrencyCode.eur]: BASIC_FORMATTERS.WESTERN,
};

const CURRENCY_FORMATTERS_WITH_SYMBOL = {
    [CurrencyCode.rub]: BASIC_FORMATTERS_WITH_SYMBOL.EASTERN,
    [CurrencyCode.uah]: BASIC_FORMATTERS_WITH_SYMBOL.EASTERN,
    [CurrencyCode.kzt]: BASIC_FORMATTERS_WITH_SYMBOL.EASTERN,
    [CurrencyCode.byn]: BASIC_FORMATTERS_WITH_SYMBOL.EASTERN,

    [CurrencyCode.usd]: BASIC_FORMATTERS_WITH_SYMBOL.WESTERN,
    [CurrencyCode.eur]: BASIC_FORMATTERS_WITH_SYMBOL.WESTERN,
};

export function getFormattedPrice(currency, whole, cents) {
    const formatter = CURRENCY_FORMATTERS[currency] || BASIC_FORMATTERS.EASTERN;

    return formatter(whole, cents);
}

export function getCurrencySymbol(currency) {
    return CURRENCY_SYMBOLS[currency];
}

export function getCurrencyShortForm(currency) {
    const symbol = getCurrencySymbol(currency);

    if (symbol) {
        return symbol;
    }

    return currencyAbbrKeyset.hasKey(currency)
        ? currencyAbbrKeyset(currency)
        : currency;
}

/**
 * Форматирует валюту. Используется в formatPrice, напрямую использоваться не должна.
 * @param {string} currency Трёхбуквенный код валюты
 * @param {string} whole Целая часть валюты (рубли, доллары)
 * @param {string} cents Дробная часть валюты (копейки, центы)
 * @return {string}
 */
export default function formatCurrency(currency, whole, cents) {
    const shortForm = getCurrencyShortForm(currency);
    const formatter =
        CURRENCY_FORMATTERS_WITH_SYMBOL[currency] ||
        BASIC_FORMATTERS_WITH_SYMBOL.EASTERN;

    return formatter(whole, cents, shortForm);
}
