/*
Вернет обертку над функцией, которая будет вызываться только после того, как пройдет "delay" ms с момента последнего вызова.
Можно вызвать метод "flush" для того, чтобы функция вызвалась немедленно, не дожидаясь таймаута, при условии, что этот
таймаут запущен, то есть уже была вызвана обертка, но не вызвана целевая функция.
*/
export default function getThrottle<T extends (...args: any[]) => any>(
    func: T,
    delay: number,
): ((...args: Parameters<T>) => void) & {flush: () => void} {
    let throttleTimeout: undefined | number | NodeJS.Timer;
    let inTimeout = false;
    let lastArgs: Parameters<T> | undefined;

    const clearThrottleTimeout = (): void => {
        if (typeof throttleTimeout === 'number') {
            clearTimeout(throttleTimeout);
        } else if (throttleTimeout) {
            clearTimeout(throttleTimeout);
        }
    };

    const throttledFunction = (...args: Parameters<T>): void => {
        if (inTimeout) {
            clearThrottleTimeout();
        }

        inTimeout = true;
        lastArgs = args;

        throttleTimeout = setTimeout(() => {
            inTimeout = false;
            func(...args);
        }, delay);

        return;
    };

    throttledFunction.flush = (): void => {
        if (!inTimeout) {
            return;
        }

        inTimeout = false;
        clearThrottleTimeout();

        if (lastArgs) {
            func(...lastArgs);
        }
    };

    return throttledFunction;
}
