import moment from 'moment';

import {
    FilterTransportType,
    KeysetTransportType,
    TransportType,
} from '../transportType';
import Lang from '../../interfaces/Lang';
import SearchSegment from '../../interfaces/state/search/SearchSegment';
import IStateSearch from '../../interfaces/state/search/IStateSearch';
import IStateCurrencies from '../../interfaces/state/IStateCurrencies';
import ISearchContext from '../../interfaces/state/search/ISearchContext';

import {getYears} from '../seo/search';
import {humanList} from '../stringUtils';
import isAllDaysSearch from '../search/isAllDaysSearch';
import isLastochkaPage from '../search/isLastochkaPage';
import getTransportTypesFromSegments from '../segments/getTransportTypes';
import gatherMinPriceData from '../segments/gatherMinPriceData';
import getValueFromEnum from '../enum/getValueFromEnum';
import convertPrice from '../currencies/convertPrice';
import formatPrice from '../format/formatPrice';
// eslint-disable-next-line no-duplicate-imports
import {buildTransportForms} from '../transportType';

import keyset from '../../i18n/meta-search-v2';
import transportKeyset from '../../i18n/transport-types';

function getYear(context: ISearchContext): string {
    const {time, when} = context;

    return isAllDaysSearch(context)
        ? humanList(getYears(time))
        : moment(when.date).year();
}

function getTransport(segments: SearchSegment[]): string {
    const types = getTransportTypesFromSegments(segments);

    return humanList(
        types.map((type: TransportType): string => {
            let result: string;

            switch (type) {
                case TransportType.train:
                    result = transportKeyset('type-train-plural-genitive');
                    break;
                case TransportType.suburban:
                    result = transportKeyset('type-suburban-plural-genitive');
                    break;
                case TransportType.plane:
                    result = transportKeyset('type-plane-plural-genitive');
                    break;
                case TransportType.bus:
                    result = transportKeyset('type-bus-plural-genitive');
                    break;
                case TransportType.water:
                    result = transportKeyset('type-water-plural-genitive');
                    break;
            }

            return result;
        }),
    );
}

function getTransportType(
    transportType: FilterTransportType,
    segments: SearchSegment[],
): FilterTransportType {
    if (transportType === FilterTransportType.all) {
        const transportTypes = getTransportTypesFromSegments(segments);

        if (transportTypes.length === 1) {
            return transportTypes[0];
        }
    }

    return transportType;
}

export default function getSearchDescription(
    search: IStateSearch,
    language: Lang,
    currencies: IStateCurrencies,
): string {
    const {context, transportTypes, filtering, segments = []} = search;
    const {from, to, when, transportType: transportTypeFromContext} = context;

    const transportType = getTransportType(transportTypeFromContext, segments);
    const minPriceData = gatherMinPriceData(segments);
    const minPrice =
        minPriceData &&
        minPriceData.price &&
        convertPrice(minPriceData.price, currencies);
    const transportForms = buildTransportForms(transportTypes);
    const transportTypeInKey = isLastochkaPage(
        transportTypeFromContext,
        when,
        filtering,
    )
        ? KeysetTransportType.lastochka
        : getValueFromEnum(transportType, KeysetTransportType);
    const keysetParams = {
        from: from.title,
        to: to.title,
        year: getYear(context),
        transport:
            transportType === FilterTransportType.all && getTransport(segments),
        price: minPrice && formatPrice(minPrice, {round: true}),
        transportTypes: humanList(
            transportTypes.map(tType =>
                transportForms[tType].plural.nominative.toLowerCase(),
            ),
        ),
    };

    switch (transportTypeInKey) {
        case KeysetTransportType.all:
            return keyset('description-transport-all', keysetParams);
        case KeysetTransportType.bus:
            return keyset('description-transport-bus', keysetParams);
        case KeysetTransportType.plane:
            return keyset('description-transport-plane', keysetParams);
        case KeysetTransportType.train:
            if (minPrice) {
                return keyset(
                    'description-transport-train-price',
                    keysetParams,
                );
            }

            return keyset('description-transport-train', keysetParams);
        case KeysetTransportType.suburban:
            if (minPrice) {
                return keyset(
                    'description-transport-suburban-price',
                    keysetParams,
                );
            }

            return keyset('description-transport-suburban', keysetParams);
        case KeysetTransportType.lastochka:
            if (minPrice) {
                return keyset(
                    'description-transport-lastochka-price',
                    keysetParams,
                );
            }

            return keyset('description-transport-lastochka', keysetParams);
        case KeysetTransportType.water:
            return keyset('description-transport-water', keysetParams);
    }

    return keyset('default-description');
}
