import {HelmetProps} from 'react-helmet';

import Platform from '../../../interfaces/Platform';
import IState from '../../../interfaces/state/IState';
import IMetaLink from '../../../interfaces/lib/meta/IMetaLink';
import {TransportType} from '../../transportType';

import defaultMetaInformation from './default';

import {
    getAlternateLanguageLinks,
    getTouchAlternateLink,
} from '../../url/altLinks';
// eslint-disable-next-line no-duplicate-imports
import {getAvailableTransportTypes} from '../../transportType';
import {humanList} from '../../stringUtils';
import getBaseMeta from './getBaseMeta';
import getCanonicalCityUrl from '../../../lib/url/getCanonicalCityUrl';
import getCanonicalHomeUrl from '../../../lib/url/getCanonicalHomeUrl';
import getDesktopUrl from '../../../lib/url/getDesktopUrl';

import pageKeyset from '../../../i18n/page';
import transportTypesKeyset from '../../../i18n/transport-types';

export default function homeMeta(state: IState): HelmetProps {
    const {
        tld,
        language,
        platform,
        page,
        stationsGroup,
        directions,
        home,
        currentSettlement: {
            title: cityTitle,
            title_genitive: cityTitleGenitive,
            id,
        },
    } = state;
    const {isCity} = home;
    const {originUrl} = page;
    const availableTransportTypes = getAvailableTransportTypes(
        stationsGroup,
        directions,
    );
    const transportTypes = availableTransportTypes.map(type =>
        transportTypesKeyset(`type-${type}-plural-genitive`),
    );
    const types = humanList(transportTypes);
    const [hasTrains, hasPlanes] = [
        availableTransportTypes.includes(TransportType.train),
        availableTransportTypes.includes(TransportType.plane),
    ];
    let descriptionContent;

    if (isCity) {
        let stations;

        if (hasTrains) {
            if (hasPlanes) {
                stations = pageKeyset('stations-train-plane');
            } else {
                stations = pageKeyset('stations-train');
            }
        } else if (hasPlanes) {
            stations = pageKeyset('stations-plane');
        }

        const descriptionParams = {
            types,
            titleGenitive: cityTitleGenitive || cityTitle,
            stations,
        };

        descriptionContent = stations
            ? pageKeyset('description-with-stations', descriptionParams)
            : pageKeyset('description-without-stations', descriptionParams);
    } else {
        descriptionContent = pageKeyset('description-home');
    }

    const title = isCity
        ? pageKeyset('title-city-genitive', {
              titleGenitive: cityTitleGenitive || cityTitle,
          })
        : pageKeyset('title-index');

    const canonicalUrl = isCity
        ? getDesktopUrl(originUrl + getCanonicalCityUrl(id, tld, language))
        : getDesktopUrl(originUrl + getCanonicalHomeUrl(tld, language));

    const meta = getBaseMeta(title, descriptionContent, canonicalUrl);

    const link: IMetaLink[] = [
        ...getAlternateLanguageLinks({page, platform}),
        {
            rel: 'canonical',
            href: canonicalUrl,
        },
    ];

    if (platform !== Platform.mobile) {
        link.push(getTouchAlternateLink(canonicalUrl));
    }

    return {
        ...defaultMetaInformation(state),
        title,
        meta,
        link,
    };
}
