import {HelmetProps} from 'react-helmet';

import Platform from '../../../interfaces/Platform';
import IState from '../../../interfaces/state/IState';
import IMetaLink from '../../../interfaces/lib/meta/IMetaLink';

import {isIOS} from '../../os';
import defaultMetaInformation from './default';
import getSearchDescription from '../getSearchDescription';
import {searchWindowTitle} from '../searchTitle';
import notFoundTitle from '../notFoundTitle';
import isResultEmpty from '../../search/isResultEmpty';
import getJsonLdCrumbsForSearchPage from '../../url/crumble/getJsonLdCrumbsForSearchPage';
import getProductSemanticMarkup from '../../seo/getProductSemanticMarkup';
import {getCanonicalUrl} from '../../url/searchUrl';
import getDesktopUrl from '../../url/getDesktopUrl';
import {
    getTouchAlternateLink,
    getAlternateLanguageLinks,
} from '../../url/altLinks';
import isAllDaysSearch from '../../search/isAllDaysSearch';
import addNoIndexPageToMeta from '../addNoIndexPageToMeta';
import {isCity} from '../../point/pointType';
import {decodeSpecialCharacters} from '../../stringUtils';
import getBaseMeta from './getBaseMeta';

import keyset from '../../../i18n/meta-search-v2';
import pageKeyset from '../../../i18n/page';

export default function search(state: IState): HelmetProps {
    const {page, language, currencies, platform, user, tld} = state;
    const {context, segments, seoMetaInfo, filtering, archivalData} =
        state.search;

    const {from, to} = context;
    const cityToCitySearch = isCity(from) && isCity(to);
    const {country: countryFrom} = from;
    const {country: countryTo} = to;
    const isDifferentCountries =
        countryFrom && countryTo && countryFrom.code !== countryTo.code;

    const emptyResult = isResultEmpty({segments, page}) && !archivalData;
    const description =
        seoMetaInfo?.description ??
        (emptyResult
            ? keyset('default-description')
            : getSearchDescription(state.search, language, currencies));

    let title;
    const semanticData = [
        {
            type: 'application/ld+json',
            innerHTML: JSON.stringify(
                getJsonLdCrumbsForSearchPage(
                    context,
                    tld,
                    language,
                    page,
                    filtering,
                ),
            ),
        },
    ];
    const link: IMetaLink[] = [];

    // добавляем каноническую ссылку
    const canonicalUrl = getCanonicalUrl(
        context,
        tld,
        language,
        getDesktopUrl(page.originUrl),
        filtering,
        archivalData,
    );

    if (canonicalUrl) {
        link.push({
            rel: 'canonical',
            // safari в ios 11 и 12 добавляет в закладки каноническую страницу, вместо текущей.
            // Нет проверки на браузер потому что на ios клиенты в основном с safari сидят,
            // да и на функционал для пользователя каноникал не влияет
            href:
                !user.isBot && isIOS(user.os)
                    ? decodeSpecialCharacters(page.fullUrl)
                    : canonicalUrl,
        });

        if (platform !== Platform.mobile) {
            link.push(getTouchAlternateLink(canonicalUrl));
        }
    }

    link.push(
        ...getAlternateLanguageLinks({
            page,
            platform,
            path: getCanonicalUrl(
                context,
                tld,
                language,
                undefined,
                filtering,
                archivalData,
            ),
        }),
    );

    if (emptyResult) {
        const notFound = notFoundTitle({context});

        title = notFound.title;
    } else {
        title =
            seoMetaInfo?.title ??
            searchWindowTitle({
                context,
                segments,
                currencies,
                filtering,
                archivalData,
            }).title;

        const productSemanticMarkup = getProductSemanticMarkup(title, segments);

        if (isAllDaysSearch(context) && productSemanticMarkup) {
            semanticData.push({
                type: 'application/ld+json',
                innerHTML: JSON.stringify(productSemanticMarkup),
            });
        }
    }

    const meta = getBaseMeta(
        emptyResult ? pageKeyset('title-index') : title,
        description,
        canonicalUrl || `https://rasp.yandex.${tld}/`,
        emptyResult ? pageKeyset('description-home') : description,
    );

    // В связи с задачей https://st.yandex-team.ru/RASPFRONT-7837 пока что не ставим noIndex для поиска
    // где точки находятся в разных странах
    if (!canonicalUrl && !(cityToCitySearch && isDifferentCountries)) {
        // если нет канонической ссылки, то закрываем от индексации данную страницу
        addNoIndexPageToMeta(meta);
    }

    return {
        ...defaultMetaInformation(state),
        title,

        script: semanticData,
        meta,
        link,
    };
}
