import {HelmetProps} from 'react-helmet';
import moment from 'moment';
import upperFirst from 'lodash/upperFirst';

import {YEAR} from '../../date/formats';

import IState from '../../../interfaces/state/IState';
import IMetaLink from '../../../interfaces/lib/meta/IMetaLink';
import StationSubtype from '../../../interfaces/state/station/StationSubtype';
import StationEventList from '../../../interfaces/state/station/StationEventList';
import Platform from '../../../interfaces/Platform';
import StationType from '../../../interfaces/state/station/StationType';

import defaultMetaInformation from './default';
import getCanonicalStationUrl from '../../url/getCanonicalStationUrl';
import getDesktopUrl from '../../url/getDesktopUrl';
import {
    getAlternateLanguageLinks,
    getTouchAlternateLink,
} from '../../url/altLinks';
import getBaseMeta from './getBaseMeta';
import getJsonLdCrumbsForStationPage from '../../url/crumble/getJsonLdCrumbsForStationPage';
import titleHasSettlementTitle from '../../station/titleHasSettlementTitle';

import metaDefaultKeyset from '../../../i18n/meta-default';
import metaStationKeyset from '../../../i18n/meta-station';
import stationPlaneSeoKeyset from '../../../i18n/station-plane-seo';

const lastDotRegExp = /\.$/;

export default function stationMeta(state: IState): HelmetProps {
    const {station, page, tld, language, flags, platform} = state;

    const {
        titleGenitive,
        title,
        fullTitle,
        fullTitleDative,
        fullTitleGenitive,
        hasPopularTitle,
        settlement,
        id,
        type,
        currentSubtype,
        mainSubtype,
        whenDate,
        event,
        iataCode,
        terminalName,
        terminals,
        now,
    } = station;

    const terminal = terminals.find(
        terminalObj => terminalObj.name === terminalName,
    );
    const canonicalPath = getCanonicalStationUrl({
        id,
        tld,
        language,
        flags,
        type,
        subtype: currentSubtype,
        terminalName,
        mainSubtype,
        event,
    });

    const canonicalUrl = getDesktopUrl(page.originUrl) + canonicalPath;

    const link: IMetaLink[] = [{rel: 'canonical', href: canonicalUrl}];

    if (platform !== Platform.mobile) {
        link.push(getTouchAlternateLink(canonicalUrl));
    }

    link.push(
        ...getAlternateLanguageLinks({page, path: canonicalPath, platform}),
    );

    let metaTitle: string | undefined;
    let metaDescription: string | undefined;
    const year = moment.parseZone(whenDate || now).format(YEAR);
    const {title: settlementTitle} = settlement || {};

    const semanticData = [
        {
            type: 'application/ld+json',
            innerHTML: JSON.stringify(
                getJsonLdCrumbsForStationPage(
                    tld,
                    language,
                    page,
                    station,
                    flags,
                ),
            ),
        },
    ];

    switch (currentSubtype) {
        case StationSubtype.train:
            metaTitle =
                event === StationEventList.arrival
                    ? settlementTitle && hasPopularTitle
                        ? metaStationKeyset('title-train-arrival-popular', {
                              title,
                              cityTitle: settlementTitle,
                              year,
                          })
                        : metaStationKeyset('title-train-arrival', {
                              title,
                              year,
                          })
                    : settlementTitle && hasPopularTitle
                    ? metaStationKeyset('title-train-departure-popular', {
                          title,
                          cityTitle: settlementTitle,
                          year,
                      })
                    : metaStationKeyset('title-train-departure', {
                          title,
                          year,
                      });
            metaDescription = hasPopularTitle
                ? metaStationKeyset('description-train-popular', {
                      title: fullTitleGenitive || fullTitle || title,
                      year,
                  })
                : metaStationKeyset('description-train', {
                      titleDative: fullTitleDative || fullTitle || title,
                      year,
                  });
            break;
        case StationSubtype.suburban:
            metaTitle =
                settlementTitle && hasPopularTitle
                    ? metaStationKeyset('title-suburban-popular', {
                          titleGenitive:
                              fullTitleGenitive || titleGenitive || title,
                          cityTitle: settlementTitle,
                          year,
                      })
                    : metaStationKeyset('title-suburban', {
                          titleGenitive: title.replace(lastDotRegExp, ''),
                          year,
                      });
            metaDescription = metaStationKeyset('description-suburban', {
                title: fullTitleDative || fullTitle || title,
            });
            break;
        case StationSubtype.plane:
            if (terminal) {
                if (event === StationEventList.departure) {
                    metaDescription = stationPlaneSeoKeyset(
                        'description-with-terminal-departure',
                        {
                            airportTitle: title,
                            airportCode: iataCode || '',
                            terminal: terminal.name,
                            isDomestic: Boolean(terminal.isDomestic),
                            isInternational: Boolean(terminal.isInternational),
                        },
                    );

                    if (title === settlementTitle) {
                        metaTitle = stationPlaneSeoKeyset(
                            'title-with-terminal-departure-same-names',
                            {
                                airportTitle: title,
                                cityTitle: settlementTitle ?? '',
                                terminal: terminal.name,
                            },
                        );
                    } else {
                        metaTitle = stationPlaneSeoKeyset(
                            'title-with-terminal-departure-different-names',
                            {
                                airportTitle: title,
                                cityTitle: settlementTitle ?? '',
                                terminal: terminal.name,
                            },
                        );
                    }
                } else {
                    metaDescription = stationPlaneSeoKeyset(
                        'description-with-terminal-arrival',
                        {
                            airportTitle: title,
                            airportCode: iataCode || '',
                            terminal: terminal.name,
                            isDomestic: Boolean(terminal.isDomestic),
                            isInternational: Boolean(terminal.isInternational),
                            noSign:
                                !terminal.isDomestic &&
                                !terminal.isInternational,
                        },
                    );

                    if (title === settlementTitle) {
                        metaTitle = stationPlaneSeoKeyset(
                            'title-with-terminal-arrival-same-names',
                            {
                                airportTitle: title,
                                cityTitle: settlementTitle ?? '',
                                terminal: terminal.name,
                            },
                        );
                    } else {
                        metaTitle = stationPlaneSeoKeyset(
                            'title-with-terminal-arrival-different-names',
                            {
                                airportTitle: title,
                                cityTitle: settlementTitle ?? '',
                                terminal: terminal.name,
                            },
                        );
                    }
                }

                break;
            }

            if (event === StationEventList.departure) {
                metaDescription = stationPlaneSeoKeyset(
                    'description-departure',
                    {airportTitle: title, airportCode: iataCode || ''},
                );

                if (title === settlementTitle) {
                    metaTitle = stationPlaneSeoKeyset(
                        'title-departure-same-names',
                        {airportTitle: title, cityTitle: settlementTitle ?? ''},
                    );
                } else {
                    metaTitle = stationPlaneSeoKeyset(
                        'title-departure-different-names',
                        {airportTitle: title, cityTitle: settlementTitle ?? ''},
                    );
                }
            } else {
                metaDescription = stationPlaneSeoKeyset('description-arrival', {
                    airportTitle: title,
                    airportCode: iataCode || '',
                });

                if (title === settlementTitle) {
                    metaTitle = stationPlaneSeoKeyset(
                        'title-arrival-same-names',
                        {airportTitle: title, cityTitle: settlementTitle ?? ''},
                    );
                } else {
                    metaTitle = stationPlaneSeoKeyset(
                        'title-arrival-different-names',
                        {airportTitle: title, cityTitle: settlementTitle ?? ''},
                    );
                }
            }

            break;
        case StationSubtype.tablo:
            metaTitle =
                event === StationEventList.arrival
                    ? settlementTitle && hasPopularTitle
                        ? metaStationKeyset('title-tablo-arrival-popular', {
                              title,
                              cityTitle: settlementTitle,
                              year,
                          })
                        : metaStationKeyset('title-tablo-arrival', {
                              title,
                              year,
                          })
                    : settlementTitle && hasPopularTitle
                    ? metaStationKeyset('title-tablo-departure-popular', {
                          title,
                          cityTitle: settlementTitle,
                          year,
                      })
                    : metaStationKeyset('title-tablo-departure', {
                          title,
                          year,
                      });
            metaDescription = hasPopularTitle
                ? metaStationKeyset('description-tablo-popular', {
                      title: fullTitleGenitive || fullTitle || title,
                      year,
                  })
                : metaStationKeyset('description-tablo', {
                      titleDative: fullTitleDative || fullTitle || title,
                      year,
                  });
            break;
    }

    if (type === StationType.railroad && !currentSubtype) {
        metaTitle =
            metaTitle ||
            metaStationKeyset('default-title-train', {
                titleDative: fullTitleDative || fullTitle || title,
                year,
            });

        metaDescription =
            metaDescription ||
            metaStationKeyset('default-description-train', {
                titleDative: fullTitleDative || fullTitle || title,
            });
    }

    if (type === StationType.bus || type === StationType.water) {
        const stationName = fullTitle || title;
        const fullStationName =
            settlement?.title &&
            !titleHasSettlementTitle(stationName, settlement.title)
                ? `${stationName} (${settlement.title})`
                : stationName;

        switch (type) {
            case StationType.bus:
                metaTitle = metaStationKeyset('title-bus', {
                    title: fullStationName,
                    year,
                });
                metaDescription = metaStationKeyset('description-bus', {
                    title: fullStationName,
                });
                break;
            case StationType.water:
                metaTitle = metaStationKeyset('title-water', {
                    title: fullStationName,
                    year,
                });
                metaDescription = metaStationKeyset('description-water', {
                    title: fullStationName,
                });
                break;
        }
    }

    const defaultMeta = defaultMetaInformation(state);
    const description = upperFirst(
        metaDescription ?? metaDefaultKeyset('description'),
    );
    const titleContent = upperFirst(metaTitle ?? defaultMeta.title);

    const meta = getBaseMeta(titleContent, description, canonicalUrl);

    return {
        ...defaultMeta,
        title: titleContent,

        script: semanticData,
        meta,
        link,
    };
}
