import React from 'react';

import {HUMAN_DATE_WITH_YEAR, HUMAN} from '../date/formats';

import ISearchContext from '../../interfaces/state/search/ISearchContext';
import ISearchSort from '../../interfaces/state/search/ISearchSort';
import ISearchArchivalData from '../../interfaces/state/search/ISearchArchivalData';

import getRouteTitle from '../routeTitle';
import tankerParse from '../tanker/tankerParse';
import {
    getAllTypeUrl,
    getAllDaysUrl,
    getFormattedDate,
    searchArchivalH1,
} from './searchTitle';
import getNewPlanDescriptionData from './getNewPlanDescriptionData';

import keyset from '../../i18n/search-title-notfound';

interface INotFoundTitle {
    title: string;
    subtitle: React.ReactElement[] | string;
}

function getDateNotFoundTitle(
    context: ISearchContext,
    sort?: ISearchSort,
): INotFoundTitle {
    const {transportType} = context;

    const {showNewPlanDescription, newPlanDate} =
        getNewPlanDescriptionData(context);
    const {titleInfinitive, titleInflected} = getRouteTitle(context);

    const params = {
        titleInfinitive,
        titleInflected,
        date: getFormattedDate(context, keyset),
        newPlanDateWithYear: newPlanDate
            .format(HUMAN_DATE_WITH_YEAR)
            .toLowerCase(),
        newPlanDate: newPlanDate.format(HUMAN).toLowerCase(),
    };
    const subtitleKeysetName = showNewPlanDescription
        ? `subtitle-date-${transportType}-new-plan`
        : `subtitle-date-${transportType}`;

    return {
        title: keyset(`title-date-${transportType}`, params),
        subtitle: tankerParse(keyset(subtitleKeysetName, params), {
            alldays: {href: getAllDaysUrl(context, sort)},
            transport: {href: getAllTypeUrl(context, sort)},
        }),
    };
}

function getAllDaysNotFoundTitle(
    context: ISearchContext,
    sort?: ISearchSort,
): INotFoundTitle {
    const {transportType} = context;
    const params = {...getRouteTitle(context)};

    return {
        title: keyset(`title-alldays-${transportType}`, params),
        subtitle: tankerParse(
            keyset(`subtitle-alldays-${transportType}`, params),
            {transport: {href: getAllTypeUrl(context, sort)}},
        ),
    };
}

interface INotFoundTitleParams {
    context: ISearchContext;

    sort?: ISearchSort;
    archivalData?: ISearchArchivalData;
}

/**
 * Вернет объект заголовка для страницы выдачи "Ничего не найдено"
 * @param {Object} context - объект контекста
 * @param {Object} sort
 * @param {Object} archivalData - объект с информацией о том что рейсы ходили, но сейчас временно отменены
 * @return {{ title: string, subtitle: string }} title - сам заголовок, subtitle - дополнительный текст
 */
export default function notFoundTitle({
    context,
    sort,
    archivalData,
}: INotFoundTitleParams): INotFoundTitle {
    if (archivalData) {
        return searchArchivalH1({context, archivalData});
    }

    if (context.when.date) {
        return getDateNotFoundTitle(context, sort);
    }

    return getAllDaysNotFoundTitle(context, sort);
}
