import React from 'react';

import {ALL_TRANSPORT_TYPES} from '../transportType';

import DateSpecialValue from '../../interfaces/date/DateSpecialValue';
// eslint-disable-next-line no-duplicate-imports
import {FilterTransportType} from '../transportType';

// eslint-disable-next-line no-duplicate-imports
import {buildTransportForms} from '../transportType';
import {getToday} from '../date/utils';
import tankerParse from '../tanker/tankerParse';
import searchUrl from '../url/searchUrl';
import isLastochkaPage from '../search/isLastochkaPage';
import gatherMinPriceData from '../segments/gatherMinPriceData';
import insertMarkupIntoKey from '../../i18n/_utils/insertMarkupIntoKey';
import convertPrice from '../currencies/convertPrice';
import formatPrice from '../format/formatPrice';
import getRouteTitle from '../routeTitle';
import getTransportTypesFromSegments from '../segments/getTransportTypes';
import isLastochka from '../segments/isLastochka';
import {humanList} from '../stringUtils';

import Price from '../../components/Price/Price';

import searchTitleDateKeyset from '../../i18n/search-title-date';
import calendarKeyset from '../../i18n/calendar';
import searchPageTitleKeyset from '../../i18n/search-page-title';
import searchWindowTitleKeyset from '../../i18n/search-window-title-v2';
import searchPageArchivalH1 from '../../i18n/search-page-archival-h1';

export function getFormattedDate(context, keyset) {
    const {time, when} = context;

    const today = getToday({
        now: time.now,
        timezone: time.timezone,
    });

    const targetDay = getToday({
        now: when.date,
        timezone: time.timezone,
    });

    const dateFormat =
        today.year() !== targetDay.year()
            ? keyset('date-format-full')
            : keyset('date-format');

    return targetDay.format(dateFormat).toLowerCase();
}

function getPrice(segments, currencies, language, isPlainText) {
    const minPriceData = gatherMinPriceData(segments);
    const minPrice =
        minPriceData &&
        minPriceData.price &&
        convertPrice(minPriceData.price, currencies);

    if (!minPrice) {
        return null;
    }

    if (isPlainText) {
        return formatPrice(minPrice, {language, from: true});
    }

    return <Price from price={minPrice} />;
}

/**
 * Формирует списки типов транспорта в разных падежах и числах
 * Пример возвращаемого значения: {
 *     single: {
 *         nominative: 'автобус, самолет и поезд',
 *         genitive: 'автобуса, самолета и поезда',
 *         ...
 *     }
 *     plural: {
 *         nominative: 'автобусы, самолеты и поезда',
 *         genitive: 'автобусов, самолетов и поездов',
 *         ...
 *     }
 * }
 *
 * @param {Object[]} segments
 * @param {string} language
 * @param {Object} archivalData
 *
 * @return {Object}
 */
function formatTransportTypes(segments, language, archivalData) {
    let transportTypes = archivalData
        ? archivalData.transportTypes
        : getTransportTypesFromSegments(segments);

    transportTypes = ALL_TRANSPORT_TYPES.filter(type =>
        transportTypes.includes(type),
    );
    const transportForms = buildTransportForms(transportTypes);
    const cases = [
        'nominative',
        'genitive',
        'dative',
        'accusative',
        'instrumental',
        'prepositional',
    ];
    const numbers = ['single', 'plural'];
    const result = {};

    numbers.forEach(number => {
        result[number] = {};
        cases.forEach(nounCase => {
            result[number][nounCase] = humanList(
                transportTypes.map(
                    type => transportForms[type][number][nounCase],
                ),
                language,
            );
        });
    });

    return result;
}

export function getSubtitle(context) {
    const {when} = context;

    if (!when.date) {
        return '';
    }

    return getFormattedDate(context, searchTitleDateKeyset);
}

function getSearchTitleBase({
    context,
    segments,
    currencies,
    titleKeyset,
    isPlainText,
    filtering,
    needSubtitle = false,
    archivalData,
}) {
    const {
        transportType: transportTypeFromContext,
        language,
        when,
        searchNext,
    } = context;
    const currentIsLastochkaPage = isLastochkaPage(
        transportTypeFromContext,
        when,
        filtering,
    );

    const segmentsForTitle = currentIsLastochkaPage
        ? segments.filter(isLastochka)
        : segments;

    const route = getRouteTitle(context);

    const price = getPrice(segmentsForTitle, currencies, language, isPlainText);

    const transportType =
        archivalData?.canonical.transportType ?? transportTypeFromContext;
    let titleKey = currentIsLastochkaPage
        ? 'lastochka-page'
        : searchNext
        ? `${transportType}-next`
        : `${transportType}-date`;

    if (price) {
        titleKey += '-price';
    }

    const formattedTransportTypes = formatTransportTypes(
        segmentsForTitle,
        language,
        archivalData,
    );
    const titleParams = {
        route,
        price,
        transport: formattedTransportTypes,
        transportTypes: formattedTransportTypes.plural.nominative.toLowerCase(),
    };

    let title;

    if (isPlainText) {
        title = titleKeyset(titleKey, titleParams);
    } else {
        const titleParts = insertMarkupIntoKey(
            titleKeyset,
            titleKey,
            titleParams,
        );

        title = titleParts.reduce((acc, titlePart) => {
            if (typeof titlePart === 'string') {
                return [...acc, ...tankerParse(titlePart, {})];
            }

            return [...acc, titlePart];
        }, []);
    }

    return {title, ...(needSubtitle ? {subtitle: getSubtitle(context)} : {})};
}

/**
 * Вернет объект заголовка для страницы выдачи и <title> страницы
 * @param {Object} context - объект контекста
 * @param {Array} segments
 * @param {Object} currencies
 * @param {Object} filtering
 * @param {Object} archivalData
 * @return {{ title: string[], subtitle: string }} title - сам заголовок, subtitle - приписка даты
 */
export function searchTitle({
    context,
    segments,
    currencies,
    filtering,
    archivalData,
}) {
    return getSearchTitleBase({
        context,
        segments,
        currencies,
        filtering,
        titleKeyset: searchPageTitleKeyset,
        isPlainText: false,
        needSubtitle: true,
        archivalData,
    });
}

/**
 * Вернет объект заголовка для страницы выдачи на которой временно нет рейсов
 * @param {Object} context - объект контекста
 * @param {Object} archivalData
 * @return {{ title: string, subtitle: string }} title - сам заголовок, subtitle - приписка даты
 */
export function searchArchivalH1({context, archivalData}) {
    return getSearchTitleBase({
        context,
        segments: [],
        currencies: {},
        titleKeyset: searchPageArchivalH1,
        isPlainText: false,
        needSubtitle: true,
        archivalData,
    });
}

/**
 * Вернет объект заголовка для страницы выдачи и <title> страницы
 * @param {Object} context - объект контекста
 * @param {Array} segments
 * @param {Object} currencies
 * @param {Object} filtering
 * @param {Object} archivalData
 * @return {{ title: string, subtitle: string }} title - сам заголовок, subtitle - приписка даты
 */
export function searchWindowTitle({
    context,
    segments,
    currencies,
    filtering,
    archivalData,
}) {
    return getSearchTitleBase({
        context,
        segments,
        currencies,
        filtering,
        titleKeyset: searchWindowTitleKeyset,
        isPlainText: true,
        archivalData,
    });
}

export function getAllTypeUrl(context, sort) {
    return searchUrl({
        context: {
            ...context,
            transportType: FilterTransportType.all,
        },
        sort,
    });
}

export function getAllDaysUrl(context, sort) {
    return searchUrl({
        context: {
            ...context,
            when: {
                text: calendarKeyset('all-days'),
                special: DateSpecialValue.allDays,
            },
        },
        sort,
    });
}
